# ApiCore package - slices transfer documentation

The **slices transfer** allows one to form structure of the table on one service node with a database, form the data on another, and then send that data back to the first node for an import, and is not limited to that.


## Requirements

To begin with slices transfer, a correct installation of api-infusemedia are necessary both on the export and import sides. In a project without an api-infusemedia installation it is recommended to add an installation of api-infusemedia based on Lumen framework.


### How to install Lumen framework

You can go directly to the [documentation](https://lumen.laravel.com/docs/5.8) of Lumen for that.
Tl;dr: `composer create-project --prefer-dist laravel/lumen subdir`

### How to install api-infusemedia
This is covered in the [main readme](readme.md) of api-infusemedia

## Declaring structure

The full structure of the tables is declared in the database. The ApiCore package can parse `SHOW CREATE TABLE` output to determine most of necessary information about tables automatically. However, that does not include the way the tables related between each other by auto increment keys. For the merge to work properly, that information must be manually declared.

### Structure retrieval route endpoint
The package provides endpoint to retrieve a structure by its name over HTTPS. The route is `(Endpoint URL)/get-slice-structure`. The route requires fields `slice_exchange_name` and `schema_name` to be provided in the request. The former is used to search for a structure name, the latter is used for generated create table statements to target the specified database schema.

### Find structure locally by name
The `DTODefs` class of the current version (for example, `App\Api\V1_0\DTODefs` is responsible for finding a structure by its name. By default, it contains none. To add one, you need to override the method `getSliceExchangeStructure`. The structure is described by an array in the following format:

```
[
    [
        'table' => string $tableName,
        'keyName' => string $keyName,
        'remap' => [
            string $remappedColumn => string $columnOwnerTable
        ]
    ],
    ...
]
```

`'table' => string $tableName` this is a name of the table to create in the structure
`'keyName' => string $keyName` this is a primary key of that table 
`string $remappedColumn => string $columnOwnerTable` specify one such pair for each column of each other table that contains a reference to the auto-increment primary key of that table, and is to be transferred along within the same slice transfer operation.

It is like foreign keys, but only references to auto_increment columns must be mapped. The order of entries must be sorted that way, so that the table which auto_increment key must be used, must be defined before the table, which uses her key

## Retrieving structure

The structure retrieval is performed between two api-infusemedia installations by means of HTTPS request.
To make that request by the means of api-infusemedia, you need an instance of `InSegment\ApiCore\Services\Api2Api\Connection`. Its dependencies are:

- Connector, implementing `InSegment\ApiCore\Services\Api2Api\ConnectorInterface`, the recommended one is `InSegment\ApiCore\Services\Api2Api\GuzzleConnector`. To use `GuzzleConnector`, you will need to install an optional dependency: `composer require guzzlehttp/guzzle`
- Endpoint, implementing `InSegment\ApiCore\Services\Api2Api\EndpointInterface`. It is a simple class to implement interface with only two methods `getApiKey` and `getApiURL`, which you will need to write for your project. Method `getApiKey` should return the key to access the api-infusemedia on the URL, it is used like a password, as there is no proper public key handshake implemented yet. Method `getApiURL` should return base URL for routes of api-infusemedia on the Laravel/Lumen installation, you do not need to specify full URL of the specific route
- Optional Logger, an instance of `InSegment\ApiCore\Services\Logger` to register requests. This `Logger` will write files into storage/logs.

Once you have that instance, for example, in the variable `$apiConnection` you can request structure by name specified in `slice_exchange_name` into your local schema named in `schema_name`:

```
use InSegment\ApiCore\Models\SliceOperation;

...

/**
 * @var InSegment\ApiCore\Models\SliceStructure $structure
 */
$structure = $apiConnection->requestStructure([
    'slice_exchange_name' => 'some_structure_name',
    'schema_name' => 'my_tmp_schema',
]);

$operation = SliceOperation::newSlice(false);
$operation->uid = $structure->getUid();
$structure->execSql();
```

The structure named in `slice_exchange_name` option must be declared on the responding side.

By default, api-infusemedia would not allow insecure HTTP request to be used. Environment variable `API_API2API_ALLOW_UNSAFE_HTTP` controls that restriction.

## Working with temporary tables

After you have retrieved the remote structure, the **tables** are created from it in your specified database schema. They are temporary, yet not defined as temporary ones and will be visible by other processes and persist until dropped. You can work with these tables as you like. However, in the end, to be able to send these tables to another side, you will need to produce **counts** for consistency checks, and **a table with substitute identifiers**. To do this automatically, write models for use the helper class for working with these tables. Only the way with the helper class will be described.

### Substitute identifiers helper class

The helper class for these temporary tables is `InSegment\ApiCore\Services\DiffSlice\DiffSlice`. It is constructed with the `$structure` you have requested, and an enumeration of all used tables.

To prepare the table with supplementary identifiers, you also need to tell, how many identifiers (at maximum) you will need for each table. The identifiers left unused will be deleted, however, excessive ones will take space and time in the transfer procedure.
```
use InSegment\ApiCore\Services\DiffSlice\DiffSlice;

...

$amountsOfNewRows = [
    'necessities' => 123,
    'belongings' => 12,
    'surroundings' => 23
];

$diffSlice = new DiffSlice($structure, $allTables);
$diffSlice->establish($amountsOfNewRows);
```

Once that is done, you can start buffering and insert your data into the tables.

### Inserting data with models and additional helper

In this example, each table will have an Eloquent model for it and the helper class will produce specific importers for each of these Eloquent models classes.

```
use InSegment\ApiCore\Services\BufferedBuilder;
use InSegment\ApiCore\Services\DiffSlice\DiffSlice;

use App\Models\Necessities;
use App\Models\Belongings;
use App\Models\Surroundings;

...

BufferedBuilder::bufferedSession(function () use ($dataToSend, $diffSlice) {
    /**
     * @var InSegment\ApiCore\Services\DiffSlice\ImportClass<Necessities> $necessitiesImporter
     * @var InSegment\ApiCore\Services\DiffSlice\ImportClass<Belongings> $belongingsImporter
     * @var InSegment\ApiCore\Services\DiffSlice\ImportClass<Surroundings> $surroundingsImporter
     */
    $necessitiesImporter = $diffSlice->importClass(Necessities::class);
    $belongingsImporter = $diffSlice->importClass(Belongings::class);
    $surroundingsImporter = $diffSlice->importClass(Surroundings::class);
    
    ...
    $necessityId = $necessitiesImporter->add($necessitiesRow);
    ...
    $belongingId = $belongingsImporter->add($belongingsRow);
    ...
    $surroundingId = $surroundingsImporter->add($surroundingsRow);
    ...
});
```

Basically, `InSegment\ApiCore\Services\DiffSlice\ImportClass` provides method `add(array $attributes)`that you use for each of your rows, and returns a primary key. If the `$attributes` array has a primary key, the row is considered as update. Otherwise, if the table's primary key is auto increment, it will be assigned with a substitute identifier - not a real auto increment key, but an identifier, which will be replaced by an auto-increment key after the transfer is complete and the tables are merged on the other side.

### Other methods of the helper

`getUuidReserve()` get an instance of `InSegment\ApiCore\Models\UUIDReserve`, allows to get new substitute keys by method `nextId(string $table)`
`getUuidLogQuery()` a query for log of substitute keys to mark them used
`getTempTables(string $targetTable)` enumeration of insert, update and changes tables which are used for storing data in the operation
`dropTempTables()` use that once all data is exported
`incrementNewCount(string $table)` to count rows stored to insert table for the target table
`incrementWrittenCount(string $table)` to count all rows stored in insert, update and changes tables for the target table
`getCounters()` use that to get counters for transfer once all the data is inserted

## Exporting

Once you have populated the tables with data, you can dump the tables of the operation into an open resource stream.

```
use InSegment\ApiCore\Services\MysqlCommands;

...

$mysqlCommands = MysqlCommands::fromConnectionConfig();

/**
 * @var resource $exportHandle
 */
...

$structure->exportTables($mysqlCommands, $exportHandle);
```

To import that on the other side, you will also need some metadata:

```
$metadata = [
    'uid' => $UID,
    'counters' => $diffSlice->getCounters(),
];
```

After that you can drop your exported tables.

```
$diffSlice->dropTempTables();
```

*The package does not provide means to transfer the resource stream and/or metadata to the other side.*

## Importing

On the other side, as soon as you have received the data in some resource stream, and metadata for it, you can import:

```
use InSegment\ApiCore\Models\SliceOperation;
use InSegment\ApiCore\Services\SliceExchange;

...

/**
 * @var resource $importHandle - a resource to import
 * @var array $structure [
 *     [
 *         'table' => string $tableName,
 *         'keyName' => string $keyName,
 *         'remap' => [
 *             string $remappedColumn => string $columnOwnerTable
 *         ]
 *     ],
 *     ...
 * ]
 * @var array $metadata [
 *     'uid' => string $uid,
 *     'counters' => array $counters
 * ]
 */

$uid = $metadata['uid'];
$counters = $metadata['counters'];

$operation = SliceOperation::fromUID($uid);
$sliceExchange = new SliceExchange($operation, $structure);
$sliceExchange->import($counters, $importHandle);
$sliceExchange->finishCopy(
    // ON MERGE
    function () {
        ...
    },
    // POST-PROCESSING
    function () {
        ...
    }
);
```

The method `finishCopy` accepts two closures - one for merging and one for post-processing.

### On merge
This method is to be called before flushing temporary tables into real ones.

This code is executed within the same MySQL transaction as moving records from temporary to real tables so any exception will force rollback to the state before commit if the post-process step fails

In the 'On merge' stage you can also use method ```SliceExchange:::removeEntitiesInsertion``` to delete entities pending for insertion from temporary tables along with their dependencies.

```
    // ON MERGE
    function () use ($exchange) {
        ...
        $somethingInsertTable = $exchange->getTempInsertTable(Something::class);
        $somethingRealTable = (new Something)->getTable();
        $somethingToDeleteFromInsertion = DB::table($somethingInsertTable)->where(...);
        $sliceExchange->removeEntitiesInsertion($somethingToDeleteFromInsertion, $somethingRealTable);
    },
```

### Post-processing

This method is to be called on a post-process step, temporary tables are already flushed into real ones, but still not dropped.

This code is executed within the same MySQL transaction as moving records from temporary to real tables so any exception will force rollback to the state before commit if the post-process step fails
