<?php

namespace InSegment\ApiCore\Api;

use InSegment\ApiCore\Facades\DTODefs;
use InSegment\ApiCore\Traits\Loggable;
use InSegment\ApiCore\Services\Logger;

/**
 * Govern recursive actions on rules
 */
abstract class RuleAnalysis
{
    use Loggable;
 
    /**
     * A collection of conversion rules by class
     * 
     * @var array
     */
    protected $rules;
    
    /**
     * DTODefs instance. To be able to pass by reference
     * 
     * @var \InSegment\ApiCore\Facades\DTODefs
     */
    protected $dtoDefs;
    
    /**
     * Logger
     *
     * @var \InSegment\ApiCore\Services\Logger 
     */
    protected $logger;
    
    /**
     * Does the analysis write trace log
     *
     * @var bool 
     */
    protected $trace = false;
    
    /**
     * Constructor
     */
    public function __construct(&$rules)
    {
        $this->rules = &$rules;
        $this->dtoDefs = DTODefs::instance();
        
        if (($tracePath = config('api_core.trace_log_path'))) {
            $logsStorageBasePath = config('app.logs_storage_path', app('path.storage'));
            $this->setLogger(new Logger($logsStorageBasePath . DIRECTORY_SEPARATOR . $tracePath));
            $this->trace = true;
        }
    }
    
    /**
     * End trace session
     * 
     * @return null
     */
    public function endTrace()
    {
        if ($this->trace) {
            $this->closeLog();
        }
    }
    
    /**
     * Get rules for class
     * 
     * @param string $class
     * @return array|string
     */
    public function &getRules($class)
    {
        return $this->rules[$class];
    }
    
    /**
     * Get rules as attributes for SourceAttributeDownstream
     * 
     * @return array
     */
    public function getRulesAsAttributes()
    {
        $rulesAsAttributes = [];
        foreach ($this->rules as $class => $rules) {
            if (!is_array($rules)) {
                continue;
            }
            
            $rulesAsAttributes[$class] = [];
            foreach ($rules as $dotFormat => $mapping) {
                $rulesAsAttributes[$class][] = [$dotFormat, $mapping];
            }
        }
        
        return $rulesAsAttributes;
    }
    
    /**
     * Check if the rule is simple attribute
     * 
     * @param string $class
     * @return bool
     */
    public function isSimpleRule($class)
    {
        return !is_array($this->rules[$class]);
    }
    
    /**
     * Should descend the given array of parameters doing the opposite operation of ascendParams in reverse order
     * Stores array of ascended params
     * 
     * @param array $params
     * @return null
     */
    protected abstract function descendParams(array $params);
    
    /**
     * Should ascend the given array of parameters doing the opposite operation of descendParams in reverse order
     * Restores values of ascended params
     * 
     * @param array $params
     * @return null
     */
    protected abstract function ascendParams(array $params);
}
