<?php

namespace InSegment\ApiCore\Api;

use Closure;
use Illuminate\Support\Arr;
use InSegment\ApiCore\Interfaces\StaticRuleInspection;

/**
 * Performs some preliminary analysis on rules by a set of inspections
 */
class StaticRuleAnalysis extends RuleAnalysis
{
    /**
     * Address to current parentClass based on rules up the tree
     * 
     * @var string 
     */
    protected $address;
    
    /**
     * dotFormat stack (for inspection purposes only)
     * 
     * @var array
     */
    protected $stack = [];
    
    /**
     * Stack of ascended properties
     *
     * @var array 
     */
    protected $ascends = [];
    
    /**
     * Current parent class, which rule set is being analyzed
     * 
     * @var string 
     */
    protected $parentClass;
    
    /**
     * Current rule set (array of Mappings by dotFormat)
     *
     * @var array 
     */
    protected $ruleset;
    
    /**
     * Inspection that puts description of relations into an array
     * 
     * @param array $out
     * @return \Closure
     */
    public static function relationTypesInspection(array &$out)
    {
        /** @var StaticRuleAnalysis $this **/
        return function ($dotFormat, &$mapping, $handlerClass, $relatedClass) use (&$out) {
            return $handlerClass
                ? !!$handlerClass
                    ::enumerateRelationType($out, $this->address, $this->parentClass, $dotFormat, $mapping)
                : false;
        };
    }
    
    /**
     * Inspection that flattens relation structure for eager load
     * 
     * @param array $out
     * @return \Closure
     */
    public static function eagerLoadsInspection(array &$out)
    {
        $inspectData = [];
        /** @var StaticRuleAnalysis $this **/
        return function ($dotFormat, &$mapping, $handlerClass, $relatedClass) use (&$out, &$inspectData) {
            if ($handlerClass && $relatedClass && ($relationName = $handlerClass::getRelationName($this->parentClass, $mapping))) {
                $relationPath = isset($inspectData['address'][$this->address])
                    ? "{$inspectData['address'][$this->address]}.{$relationName}" 
                    : $relationName;
                
                $downAddress = $this->formAddress($dotFormat);
                $inspectData['address'][$downAddress] = $relationPath;
                $table = (new $relatedClass)->getTable();
                
                $parentAlwaysNew = !empty($inspectData['always-new'][$this->address]);
                $currentAlwaysNew = in_array('always-new', Arr::get($mapping, 'extra', []));
                
                if ($parentAlwaysNew || $currentAlwaysNew) {
                    $out['relations']['export'][] = &$relationPath;
                    $inspectData['always-new'][$downAddress] = true;
                } else {
                    $out['relations']['all'][] = &$relationPath;
                }
                
                $out['tables'][$table][] = &$relationPath;
                    
                return $handlerClass::importsRelation($mapping);
            }
            
            return false;
        };
    }
    
    /**
     * Inspection that enumerates table enabled for writing with Transactor
     * 
     * @param array $out
     * @param bool $withReadOnly
     * @return \Closure
     */
    public static function enabledToWriteInspection(array &$out, bool $withReadOnly = false)
    {
        !isset($out['enabled']) && ($out['enabled'] = []);
        !isset($out['key-name']) && ($out['key-name'] = []);
        
        /** @var StaticRuleAnalysis $this **/
        return function ($dotFormat, &$mapping, $handlerClass, $relatedClass) use (&$out, $withReadOnly) {
            if ($handlerClass && $relatedClass) {
                $hasOutput = isset($out['enabled'][$relatedClass]);
                $readOnlyCondition = $withReadOnly || !in_array('read-only', Arr::get($mapping, 'extra', []));

                if (!$hasOutput && $readOnlyCondition) {
                    $example = new $relatedClass;
                    $table = $example->getTable();
                    $pk = $example->getKeyName();
                    $out['enabled'][$relatedClass] = $table;
                    $out['key-name'][$table] = $pk;
                    
                    if (isset($mapping['addWith'])) {
                        foreach ($mapping['addWith'] as $addWithClass => $addWith) {
                            if (is_int($addWithClass)) {
                                $addWithClass = Arr::wrap($addWith)[0];
                            }
                            
                            $addWithExample = new $addWithClass;
                            $addWithTable = $addWithExample->getTable();
                            $addWithPk = $addWithExample->getKeyName();
                            
                            if (!isset($out['enabled'][$addWithClass])) {
                                $out['enabled'][$addWithClass] = $addWithTable;
                                $out['key-name'][$addWithTable] = $addWithPk;
                            }
                        }
                    }
                    
                    if ($handlerClass::isDeletionEnabled($mapping)) {
                        $out['deletable'][$table] = true;
                    }
                    
                    return true;
                }
            }
            
            return false;
        };
    }

    /**
     * Inspection that determines concrete auto-count inspector by handler and delegates work to him
     * 
     * @param array $out
     * @param array $inspectData
     * @return \Closure
     */
    public static function autoCountInspection(array &$out, array &$inspectData)
    {
        $inspectorByHandler = [];
        /** @var StaticRuleAnalysis $this **/
        return function ($dotFormat, &$mapping, $handlerClass, $relatedClass) use (&$out, &$inspectData, &$inspectorByHandler) {
            if ($handlerClass) {
                /** @var \InSegment\ApiCore\Inspections\AutoCountInspection $inspector **/
                if (isset($inspectorByHandler[$handlerClass])) {
                    $inspector = $inspectorByHandler[$handlerClass];
                } else {
                    $inspectorClass = $handlerClass::getAutoCountInspectionClass();
                    $inspector = new $inspectorClass($this, $out, $inspectData);
                }
                
                return $inspector->inspectRule($dotFormat, $mapping, $handlerClass, $relatedClass);
            } else {
                return false;
            }
        };
    }
    
    /**
     * Constructor
     * Accepts dictionary of class rule sets, class name of the root class, and array of callable inspection methods
     * If the inspection should not proceed to further relations, it should return false and be removed
     * 
     * Closure inspection methods will have $this set to the instance of StaticRuleAnalysis
     * Also accepts class names of StaticRuleInspection and makes instance of them automatically
     * 
     * @param array $rules
     * @param string $rootClass
     * @param array $methods
     */
    public function __construct(&$rules, $rootClass)
    {
        parent::__construct($rules);
        $this->address = (new $rootClass)->getTable();
        $this->parentClass = $rootClass;
        $this->ruleset = &$this->getRules($rootClass);
        $this->methods = [];
    }
    
    /**
     * Set methods of inspection
     * 
     * @param array $methods
     * @return $this
     */
    public function setMethods($methods)
    {
        $this->methods = array_map(function ($method) {
            if ($method instanceof Closure) {
                return $method->bindTo($this, static::class);
            } else if (is_string($method)) {
                return (new $method($this));
            } else {
                return $method;
            }
        }, $methods);
        
        return $this;
    }
    
    /**
     * Getter for $this->address
     * 
     * @return string
     */
    public function getAddress()
    {
        return $this->address;
    }
    
    /**
     * Getter for $this->parentClass
     * 
     * @return string
     */
    public function getParentClass()
    {
        return $this->parentClass;
    }
    
    /**
     * Getter for $this->stack
     * 
     * @return array
     */
    public function getStack()
    {
        return $this->stack;
    }
    
    /**
     * Should descend the given array of parameters doing the opposite operation of ascendParams in reverse order
     * Stores array of ascended params
     * 
     * @param array $params
     * @return null
     */
    protected function descendParams(array $params)
    {
        $ascended = [];
        
        if (isset($params['parentClass'])) {
            $ascended['parentClass'] = $this->descendParentClass($params['parentClass']);
        }
        
        if (isset($params['dotFormat'])) {
            $ascended['dotFormat'] = $this->descendDotFormat($params['dotFormat']);
        }
        
        if (isset($params['methods'])) {
            $ascended['methods'] = $this->descendMethods($params['methods']);
        }
        
        $this->ascends[] = $ascended;
    }
    
    /**
     * Should ascend the given array of parameters doing the opposite operation of descendParams in reverse order
     * Restores values of ascended params
     * 
     * @param array $params
     * @return null
     */
    protected function ascendParams(array $params)
    {
        $ascended = array_pop($this->ascends);
        
        if (isset($params['methods'])) {
            $this->ascendMethods($ascended['methods']);
        }
        
        if (isset($params['dotFormat'])) {
            $this->ascendDotFormat($ascended['dotFormat']);
        }
        
        if (isset($params['parentClass'])) {
            $this->ascendParentClass($ascended['parentClass']);
        }
    }
    
    /**
     * Analyze rules with methods
     * 
     * @return null
     */
    public function performAnalysis()
    {
        return ($func = function () use (&$func) {
            if (is_array($this->ruleset)) {
                foreach (($ruleset = $this->ruleset) as $dotFormat => $mapping) {
                    $handler = is_array($mapping) ? $this->dtoDefs->classifyMapping($mapping) : null;
                    $class = isset($handler) ? $handler::getRelatedClass($mapping) : null;

                    // each method executed and determines its applicability, and if not applicable,
                    // then it should not be called on relations down the tree
                    $methods = $this->methods;
                    foreach ($this->methods as $key => $inspection) {
                        $bool = $inspection instanceof StaticRuleInspection
                            ? $inspection->inspectRule($dotFormat, $mapping, $handler, $class)
                            : $inspection($dotFormat, $mapping, $handler, $class);
                        
                        if (!$bool) {
                            unset($methods[$key]);
                        }
                    }
                    
                    if (count($methods) != count($this->methods)) {
                        $methods = array_values($methods);
                    }
                    
                    // branch into relations down the tree recursive
                    if (isset($class) && $methods) { 
                        $params = ['parentClass' => $class, 'methods' => &$methods, 'dotFormat' => $dotFormat];
                        $this->descendParams($params);
                        $func();
                        $this->ascendParams($params);
                    }
                }
            }
        })();
    }
    
    /**
     * Descend methods
     * 
     * @param array $methods
     * @return array
     */
    protected function &descendMethods(&$methods)
    {
        $ascendedMethods = &$this->methods;
        $this->methods = &$methods;
        return $ascendedMethods;
    }
    
    /**
     * Ascend methods
     * 
     * @param array $ascendedMethods
     * @return null
     */
    protected function ascendMethods(&$ascendedMethods)
    {
        $this->methods = &$ascendedMethods;
    }
    
    /**
     * Descend dotFormat, special case is dotFormat === '-'
     * Ascendence of dotFormat is stored into stack
     * 
     * @param string $dotFormat
     * @return string
     */
    protected function descendDotFormat($dotFormat)
    {
        $ascended = ['address' => $this->address, 'dotFormat' => $dotFormat];
        array_push($this->stack, $dotFormat);
        $this->address = $this->formAddress($dotFormat);
        return $ascended;
    }
    
    /**
     * Forms new full address consisting of current address and specified dotFormat
     * 
     * @param string $dotFormat
     * @return string
     */
    protected function formAddress($dotFormat)
    {
        return $dotFormat === '-'
            ? $this->address
            : ($this->address
                ? implode('.', [$this->address, $dotFormat])
                : $dotFormat
            );
    }
    
    /**
     * Ascend dotFormat,
     * 
     * @param array $ascended
     * @return null
     */
    protected function ascendDotFormat(&$ascended)
    {
        $this->address = $ascended['address'];
        
        if (isset($ascended['dotFormat']) && array_pop($this->stack) !== $ascended['dotFormat']) {
            throw new \Exception("Something went wrong! Descendance stack was affected!");
        }
    }
    
    /**
     * Descend parentClass
     * Also loads different rule sets based on classes
     * 
     * @param string $parentClass
     * @return array
     */
    protected function descendParentClass(&$parentClass)
    {
        $ascended = ['parentClass' => $this->parentClass, 'ruleset' => $this->ruleset];
        $this->parentClass = $parentClass;
        $this->ruleset = &$this->getRules($parentClass);
        return $ascended;
    }
    
    /**
     * Ascend parentClass and related rule set
     * 
     * @param array $ascencedProps
     * @return null
     */
    protected function ascendParentClass(&$ascencedProps)
    {
        $this->parentClass = &$ascencedProps['parentClass'];
        $this->ruleset = &$ascencedProps['ruleset'];
    }
    
}
