<?php

namespace InSegment\ApiCore\Api\Concerns;

use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use InSegment\ApiCore\Facades\DTODefs;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

use InSegment\ApiCore\Api\DataTransferObject;

trait ExportsObjects
{
    /**
     * Export one or a set of Models
     * 
     * @param \Illuminate\Database\Eloquent\Collection|array|\Illuminate\Database\Eloquent\Model $models
     * @param array|string|null $conversion
     * @param bool $useContext
     * @return $this
     */
    public function export($models, &$conversion = null, bool $useContext = false)
    {
        if (is_array($models)) {
            $models = collect($models);
        }

        $context = $useContext ? DataTransferObject::CONTEXT_EXPORT : null;
        
        if ($models instanceof Collection) {
            foreach ($models as $dotFormat => $model) {
                $this->exportModel(['model' => $model, 'dotFormat' => $dotFormat, 'context' => $context, 'omit' => true], $conversion);
            }
        } else {
            $this->exportModel(['model' => $models, 'context' => $context, 'omit' => true], $conversion);
        }
        
        return $this;
    }
    
    /**
     * Export fields
     * Codes the Model data back to its original mappings
     * 
     * @param array $params
     * @param array|string|null $conversion - only this subset of base rules will apply
     * @return null
     */
    public function exportModel(array $params, $conversion = null)
    {
        $this->descendParams($params);

        $convertedRules = $this->dtoDefs->rulesByConversion($conversion, $this->getRules($this->modelClass));
        $context = DataTransferObject::CONTEXT_EXPORT;
        
        if (!is_array($convertedRules)) {
            $this->input = $this->model->$convertedRules;
        } else {
            foreach ($convertedRules as $dotFormat => $mapping) {
                if (empty($mapping) || isset($this->omit[$dotFormat]) || isset($mapping['auto'])) {
                    continue;
                }
                
                $this->exportMapping(['dotFormat' => $dotFormat, 'mapping' => $mapping, 'context' => $context]);
        
                if (Arr::get($this->input, $dotFormat) === null) {
                    Arr::forget($this->input, $dotFormat);
                }
            }
        }
        
        if (!empty($this->context)) {
            foreach (array_keys($this->context) as $dotFormat) {
                $value = Arr::get($this->input, $dotFormat);
                
                if (isset($value)) {
                    $this->context[$dotFormat] = $value;
                }
            }
        }
        
        $this->ascendParams($params);
    }
    
    /**
     * Export mapping
     * 
     * @param array $params
     * @return null
     */
    protected function exportMapping(array $params)
    {
        $this->descendParams($params);
        
        if (is_array($this->mapping)) {
            $typeFromMapping = Arr::get($this->mapping, 'type');

            if ($typeFromMapping && isset($this->mappers[$typeFromMapping])) {
                $this->mappers[$typeFromMapping]->export($this->mapping);
            }
        } else {
            $this->exportMappedField();
        }
        
        $this->ascendParams($params);
    }
    
    /**
     * Export field mapping, supporting calling and casting, and dot notation
     * 
     * @return null
     */
    private function exportMappedField()
    {
        $target = $this->model;
        foreach (explode('.', $this->mapping) as $part) {
            $isFun = Str::endsWith($part, '()');
            if ($isFun) {
                $part = substr($part, 0, -2);
            }
            
            $explode = explode(':', $part, 2);
            $part = $explode[0];
            $cast = $explode[1] ?? null;
            
            if (!isset($target)) {
                return;
            }
            
            if (isset($cast)) {
                if (($isFun = Str::endsWith($part, '()'))) {
                    $part = substr($part, 0, -2);
                    $target = $target->$part();
                } else {
                    $target = data_get($target, $part);
                }
                
                switch($cast) {
                    case 'array': $target = (array) $target; break;
                    case 'string': $target = (string) $target; break;
                    case 'double': $target = (double) $target; break;
                    case 'float': $target = (float) $target; break;
                    case 'int': $target = (int) $target; break;
                }
            } else if ($isFun) {
                $target = $target->$part();
            } else {
                $target = data_get($target, $part);
            }
        }
        
        if (isset($target)) {
            $this->input = $target;
        }
    }
    
    /**
     * Export relation mapping
     * 
     * @param array $mapping
     * @return null
     */
    private function exportRelation(&$mapping)
    {
        $relation = $this->model->{$mapping['relation']}();
        $related = $this->model->{$mapping['relation']};
        
        if ($relation instanceof BelongsTo && is_string($externalKeyName = $this->externalKeyName($related, $mapping))) {
            $this->input = $related->$externalKeyName;
        } else if ($related instanceof Collection) {
            $context = DataTransferObject::CONTEXT_EXPORT;
            foreach ($related as $dotFormat => $relatedModel) {
                $params = ['dotFormat' => $dotFormat, 'context' => $context];
                $this->descendParams($params);
                $this->export($relatedModel);
                $this->ascendParams($params);
            }
        } else if (isset($related)) {
            $this->export($related);
        }
    }
    
    /**
     * Get external key name by related model and mapping for its relation
     * 
     * @param \Illuminate\Database\Eloquent\Model|null $related
     * @param array $mapping
     * @return string|null
     */
    private function externalKeyName($related, &$mapping)
    {
        if (!isset($related)) {
            return null;
        }
        
        if (array_key_exists('key', $mapping)) {
            return $mapping['key'];
        }
        
        return DTODefs::getExternalKeyDefaultName(get_class($related));
    }
}
