<?php

namespace InSegment\ApiCore\Api\Concerns;

use Illuminate\Support\Arr;
use InSegment\ApiCore\Interfaces\TransactibleModel;

use InSegment\ApiCore\Api\DataTransferObject;
use InSegment\ApiCore\Services\Transactor;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

trait ImportsObjects
{
    /**
     * Whether to rewrite values
     * 
     * @var bool
     */
    private $update;
    
    /**
     * Import one or a set of Models
     * 
     * @param \Illuminate\Database\Eloquent\Collection|array|\Illuminate\Database\Eloquent\Model $models
     * @param bool $useContext
     * @return $this
     */
    public function import($models, bool $useContext = false)
    {
        if (is_array($models)) {
            $models = collect($models);
        }
        
        $context = $useContext ? DataTransferObject::CONTEXT_IMPORT : null;
        
        if ($models instanceof Collection) {
            foreach ($models as $dotFormat => $model) {
                $this->importModel(['model' => $model, 'dotFormat' => $dotFormat, 'context' => $context, 'omit' => true]);
            }
        } else {
            $this->importModel(['model' => $models, 'context' => $context, 'omit' => true]);
        }
        
        return $this;
    }
    
    /**
     * Import a set of Models keyed ith external keys
     * 
     * @param \Illuminate\Database\Eloquent\Collection|array $models
     * @param string $externalKey
     * @param array $inputKeyByExtKeys
     * @param bool $useContext
     */
    public function importManyWithKeys($models, string $externalKey, array $inputKeyByExtKeys, bool $useContext = false)
    {
        if (is_array($models)) {
            $models = collect($models);
        }
        
        $context = $useContext ? DataTransferObject::CONTEXT_IMPORT : null;
        $reversedInputKeys = null;

        foreach ($models as $model) {
            // remember, new objects are last ones
            if ($model->exists) {
                $key = $model->getAttribute($externalKey);
                $dotFormat = $inputKeyByExtKeys[$key];

                // unset for found objects, as new objects are last ones, again
                unset($inputKeyByExtKeys[$key]);
            } else {
                if (!isset($reversedInputKeys)) {
                    $reversedInputKeys = array_reverse($inputKeyByExtKeys);
                }
                
                // now proceed with completely new objects (they are last)
                $dotFormat = array_pop($reversedInputKeys);
            }
            
            $this->importModel(['model' => $model, 'dotFormat' => $dotFormat, 'context' => $context, 'omit' => true]);
        }
    }
    
    /**
     * Run import scenario
     * 
     * First, we appoint transaction for the model, so it has primary key set up. This is done by default in method
     * Transactible::newFromImport, however the method could be overriden to do some other actions, to force skip import
     * or something else. Remember that, if you won't appoint transaction in advance, the primary key of the Model
     * might be unavailable to use with relations.
     * 
     * At the second step, we relate necessary Models between each other. Models which we depend on
     * (belong to, and so on) go ahead along with attributes of the current model being set. Models which depend
     * on this particular one, come next, but still before this Model is written.
     * 
     * In past, the Model was written before relating Models which depended on the current one. But then I've added
     * an 'inverseOf' relation mechanism Yii2-like, and TransferImporter uses this feature. So each related Model
     * depending on this particular one via relation with specified inverseOf will be able to affect its parent
     * without querying the database, and so will use our existing instance and no need to write it preliminary.
     * 
     * Finally, the Model is written.
     * 
     * @see \InSegment\ApiCore\Traits\Transactible
     * @see \InSegment\ApiCore\Interfaces\TransactibleModel
     * @see \InSegment\ApiCore\Services\BufferedBuilder
     * @param array $params
     * @return $this
     */
    protected function importModel(array $params)
    {
        $this->descendParams($params);
        
        // first step
        if (!$this->model instanceof TransactibleModel) {
            throw new \Exception("Any Model going through import must implement TransactibleModel contract");
        }
        
        $this->model = $this->model->newFromImport($this->input, $this->context, $this->update);
        
        // second step
        if ($this->model) {
            $this->mapAttributes()->relateDescendants();
            $this->model->save([], false, $this->relationToModel);
        }
        
        $this->ascendParams($params);
        return $this;
    }
    
    /**
     * An entry point for actual value conversion and relation. Not a "small method" at all.
     * Call to $this->descend and then mappingToAttrubute makes way too much important things
     * 
     * @return $this
     */
    protected function mapAttributes()
    {
        $classRules = &$this->getRules($this->modelClass);
        
        // This actually maps received JSON objects to fields on $this Model
        if (empty($classRules)) {
            throw new \Exception("No rules found in JSON for the model: '{$this->modelClass}'");
        } else if (is_array($classRules)) {
            foreach ($classRules as $dotFormat => &$mapping) {
                if (empty($mapping) || isset($this->omit[$dotFormat])) {
                    continue;
                }
                
                if (isset($mapping['auto'])) {
                    data_fill($this->input, $dotFormat, $mapping['auto']);
                }
                
                if ($dotFormat === '-' || (is_array($this->input) && Arr::has($this->input, $dotFormat))) {
                    $this->mappingToAttribute(['dotFormat' => $dotFormat, 'mapping' => $mapping]);
                }
            }
        } else {
            // Map object's field to a one-value record
            $this->mappingToAttribute(['mapping' => $classRules, 'context' => DataTransferObject::CONTEXT_IMPORT]);
        }
        
        return $this;
    }
    
    /**
     * This method's purpose is to set up the field of the Model from an object
     * It parses a rule and performs an action
     * 
     * @param array $params
     * @return $this
     */
    protected function mappingToAttribute(array $params)
    {
        $this->descendParams($params);
        
        if (is_array($this->mapping)) {
            $typeFromMapping = Arr::get($this->mapping, 'type');

            if ($typeFromMapping && isset($this->mappers[$typeFromMapping])) {
                $this->mappers[$typeFromMapping]->import($this->mapping);
            }
        } else if ($this->update || !$this->model->getAttribute($this->mapping)) {
            $this->model->setAttribute($this->mapping, $this->input);
        }
        
        $this->ascendParams($params);
        return $this;
    }
    
    /**
     * This method descends input by the rules of relation from this Model to another.
     * If the relation is having a field of this Model filled with an id (or some other key) of another Model,
     * the another Model will be now processed.
     * 
     * Otherwise, the relation and a pointer to the input will be stored for later
     * to the stack of "downStream" relations.
     * 
     * @return $this
     * @throws \Exception
     */
    protected function relateToAttribute() 
    {
        if (!isset($this->input)) {
            return;
        }
        
        /**
         * If this is a downstream relation, this call is useless, since we may not have proper fields yet
         * But, if it is upstream relation, it is useful right now and we have no totally chance to call it later
         */
        $relation = $this->model->{$this->mapping['relation']}();
        $extra = Arr::get($this->mapping, 'extra', []);
        $alwaysNew = in_array('always-new', $extra);
        
        if ($relation instanceof BelongsTo) {
            $this->relateBelongsToRelation(['context' => DataTransferObject::CONTEXT_IMPORT], $relation, $alwaysNew);
        } else {
            if ($this->mapping['type'] === 'find') {
                throw new \Exception("Find type of relation can only be used with BelongsTo");
            }
            
            $fresh = in_array('fresh', $extra);
            $searchMode =
                ($fresh ? DataTransferObject::SEARCH_MODE_FRESH : null)
                ?? ($alwaysNew ? DataTransferObject::SEARCH_MODE_NEW : null)
                ?? DataTransferObject::SEARCH_MODE_NORMAL;
            
            // if it is upstream relation, set data for its processing later in part2
            $this->modelDownStream[] = [$this->mapping, last($this->stack), $searchMode];
        }
    }
    
    /**
     * Relate BelongsTo relation. Handles cases such is 'only id given',
     * found or not found, import if given the whole object, et cetera...
     * 
     * @param array $params
     * @param \Illuminate\Database\Eloquent\Relations\Relation $relation
     * @param bool $alwaysNew
     * @throws \Exception
     */
    protected function relateBelongsToRelation(array $params, $relation, $alwaysNew)
    {
        $this->descendParams($params);

        $relationName = $this->mapping['relation'];
        $typeFromMapping = $this->mappers[$this->mapping['type']];
        $relatedClass = $typeFromMapping->getRelatedClass($this->mapping);
        $extKeysInfo = $typeFromMapping->getExtKeysInfo($this->dtoDefs, $this->mapping, $this->input);

        // Use eager load if possible, ignore eager load when the model should always be new
        if (!$alwaysNew && $this->model->relationLoaded($relationName)) {
            $related = $this->model->getRelation($relationName);
        }

        // state in that variable, whether we don't have the model from eager load
        // eager loaded models should avoid checks on DataTransferObject::DATA_GIVEN_NOT_ENOUGH,
        // because they are not affected by input
        $findRelated = !isset($related);

        if ($findRelated) {
            $related = $this->findOrMakeChild($relation, $this->mapping, $extKeysInfo);
        }

        $dataCharacteristic = $this->dtoDefs->characterizeGivenData($relatedClass, $this->input, $related->exists, $extKeysInfo);

        // $findRelated means that model didn't come from eager load
        // if there are ext keys specified in rules mapped to values present or absent in input
        // if characterized as not enough data and do not exists already
        if ($findRelated && count($extKeysInfo) && $dataCharacteristic === DataTransferObject::DATA_GIVEN_NOT_ENOUGH && !$related->exists) {
            $debug = json_encode([
                'keysInfo' => $extKeysInfo,
                'rules' => $this->rules[$relatedClass],
                'mapping' => $this->mapping,
                'input' => $this->input
            ], JSON_PRETTY_PRINT|JSON_PARTIAL_OUTPUT_ON_ERROR);

            throw new \Exception("No data is given for a key of {$relatedClass} which is not present in the database or in the transaction. Debug: {$debug}");
        }

        // Import the object like any other if the create/update data is present, that means conditions:
        // 1. mapping imports relations ('find' for example does not)
        // 2. $related does not exists OR the input has more data than DATA_GIVEN_NOT_ENOUGH or DATA_GIVEN_KEY_NO_CONTENT
        if ($typeFromMapping->importsRelation($this->mapping)) {
            $byCharachteristic = !in_array($dataCharacteristic, [DataTransferObject::DATA_GIVEN_KEY_NO_CONTENT, DataTransferObject::DATA_GIVEN_NOT_ENOUGH]);
            $byNotExists = !$related->exists;

            if ($byCharachteristic || $byNotExists) {
                // TODO: Add specific handling for MorphTo Relations
                // When a Model is always new, there may be contradiction on relation on its primary key, avoid it
                $query = $relation->getQuery()->getQuery();
                $keyName = $relation->getQualifiedOwnerKeyName();
                foreach($query->wheres as &$value) {
                    if ($value['column'] === $keyName) {
                        if ($alwaysNew && $value['type'] === 'Basic' && $value['operator'] === '=') {
                            $value['operator'] = '!=';
                        } else if ($findRelated && $value['type'] === 'Null') {
                            $value['type'] = 'NotNull';
                        }
                        break;
                    }
                }

                $this->importModel(['model' => $related, 'relation' => $relation, 'omit' => true]);
            }
        }

        // Actually, sets $this->field_in_question to the key of the $related model
        $relation->associate($related);
        $this->model->setRelation($relationName, $related);

        $this->ascendParams($params);
    }
    
    /**
     * After the Model is saved, we must return to the relations pointing "down" the relation tree.
     * These relations are such when a field with a parent's id (or some other key) is stored in the child's table
     * 
     * @return $this
     */
    protected function relateDescendants()
    {
        $downStream = $this->modelDownStream;
        foreach ($downStream as &$downInfo) {
            list($mapping, $dotFormat, $searchMode) = $downInfo;
            $relationName = $mapping['relation'];
            $type = $mapping['type'];

            $relation = $this->model->$relationName();
            
            if ($type == 'one') {
                $context = DataTransferObject::CONTEXT_IMPORT;
                $model = $this->dtoDefs->getOneRelated($this->model, $relationName, $searchMode);
                $this->model->setRelation($relationName, $model);
                $this->importModel(['relation' => $relation, 'model' => $model, 'dotFormat' => $dotFormat, 'mapping' => $mapping, 'context' => $context, 'omit' => true]);
            } else if ($type == 'many') {
                $this->importDescendants(['relation' => $relation, 'dotFormat' => $dotFormat, 'mapping' => $mapping], $relationName, $searchMode);
            }
        }
        
        return $this;
    }
    
    /**
     * Import Models which are dependent on the currently previous-in-stack Model
     * 
     * @throws \Exception
     * @param array $params
     * @param string $relation
     * @param int $searchMode
     * @return $this
     */
    protected function importDescendants(array $params, string $relation, int $searchMode)
    {
        $this->descendParams($params);
        
        if (!is_array($this->input)) {
            throw new \Exception('Value is not an array on relation with many');
        }

        // clone relation to avoid setting wrong restrictions on original relation object
        $relatedExample = $this->relationToModel->getRelated();
        $relatedClass = get_class($relatedExample);
        
        // if the external key is defined, we can get items in any order, with any gaps
        if (($externalKey = $this->dtoDefs->getExternalKeyDefaultName($relatedClass))) {
            $relInfo = $this->dtoDefs->mapKeysByInput($relatedClass, $this->input) + ['relation' => $relation, 'searchMode' => $searchMode, 'externalKey' => $externalKey];
            
            // get existing and new objects, new objects are last ones
            $relatedPlusNew = $this->dtoDefs->getManyRelatedWithKeys($this->model, $relInfo, count($this->input));
            
            $this->model->setRelation($relation, $relatedPlusNew);
            
            // each found object certainly have data for it, since it is joined with buffer
            $this->importManyWithKeys($relatedPlusNew, $externalKey, $relInfo['inputKeyByExtKeys'], true);
        } else {
            // otherwise work in the correct order
            // get maximum of so much records, that we've received
            
            $relatedPlusNew = $this->dtoDefs->getManyRelated($this->model, $relation, count($this->input), $searchMode);
            
            $this->model->setRelation($relation, $relatedPlusNew);
            
            // clone relation to avoid setting wrong restrictions on original relation object
            /* @see EloquentServiceProvider */
            $this->import($relatedPlusNew);
        }
        
        $this->ascendParams($params);
        return $this;
    }
    
    /**
     * This method works with BelongsTo relations. It is different from Has* relations by the means of
     * creating the related model before the parent model is saved, and associating it to the unsaved parent.
     * The default expected usage is:
     * given id, find the model or create if not found and associate it with parent;
     * given an object, find the model or create it by id from object, set necessary fields, and associate it with parent;
     * 
     * @param \Illuminate\Database\Eloquent\Relations\BelongsTo $relation
     * @param array $mapping
     * @param bool $onlyId
     * @throws \Exception
     */
    private function findOrMakeChild(BelongsTo $relation, array &$mapping, &$extKeysInfo)
    {
        $typeFromMapping = $this->mappers[$mapping['type']];
        $relatedClass = $typeFromMapping->getRelatedClass($mapping);
        
        // If $noSearch is true, there will be no query for existing Model, new one will be created instead
        $noSearch = in_array('always-new', Arr::get($mapping, 'extra', []));
        $command = $noSearch ? 'newModelInstance' : 'firstOrNew';
        
        $query = clone $relation;
        $baseQuery = $query->getBaseQuery();
        $baseQuery->wheres = [];
        $baseQuery->bindings['where'] = [];
        
        if (count($extKeysInfo)) {
            $suppliesForUniques = [];
            if (isset($mapping['uniqueSupply'])) {
                $this->exportModel(['input' => &$suppliesForUniques, 'dotFormat' => '-', 'omit' => true], $mapping['uniqueSupply']);
            }
            
            $searchAttributes = $this->dtoDefs->uniquesToAttributes($suppliesForUniques, $relatedClass, $extKeysInfo);
            
            $related = $this->getFromPreload($relatedClass, $extKeysInfo, $searchAttributes);
            if (!isset($related)) {
                $related = $query->$command($searchAttributes);
            }
            
            return $related;
        } else {
            $checkable = $this->dtoDefs->collectCheckableAttributes($relatedClass, $this->input);
            return $query->$command($checkable);
        }
    }
    
    
    /**
     * Get Model from preload, if it is there
     * 
     * @param string $class
     * @param array $extKeysInfo
     * @param array $searchAttributes
     * @return \Illuminate\Database\Eloquent\Model|null
     */
    private function getFromPreload($class, &$extKeysInfo, &$searchAttributes)
    {
        !isset($this->preloads[$class]) && ($this->preloads[$class] = []);
        $extKeysImplode = implode(',', array_keys($extKeysInfo));
        
        if (!array_key_exists($extKeysImplode, $this->preloads[$class])) {
            $preloadClosure = Transactor::getPreloads($this->modelClass, $class, $extKeysImplode);
            $this->preloads[$class][$extKeysImplode] = isset($preloadClosure) ? $preloadClosure() : null;
        }
        
        if (isset($this->preloads[$class][$extKeysImplode])) {
            if (($group = $this->preloads[$class][$extKeysImplode]->get(implode(',', array_values($extKeysInfo))))) {
                foreach ($searchAttributes as $key => $value) {
                    $group = $group->where($key, $value);
                }
                
                return $group->first();
            }
        }
        
        return null;
    }
    
}