<?php

namespace InSegment\ApiCore\Api\V_1_0;

use Illuminate\Database\Eloquent\Collection;

use Illuminate\Support\Arr;
use InSegment\ApiCore\Api\DataTransferObject;
use InSegment\ApiCore\Api\StaticRuleAnalysis;

use InSegment\ApiCore\Middleware\ChooseVersion;
use InSegment\ApiCore\Inspections\PreloadInspection;

use InSegment\ApiCore\Models\UUIDGeneration;
use InSegment\ApiCore\Models\LogReceivedRecord;
use InSegment\ApiCore\Models\DeletedTransactionRecord;
use InSegment\ApiCore\Traits\BuildsDictionaries;

use InSegment\ApiCore\Services\Transactor;
use InSegment\ApiCore\Services\ParseCreateTable;

use InSegment\ApiCore\Interfaces\DTOCharacteristics;

use Illuminate\Support\Facades\Cache;

abstract class DTOUtil
{
    use BuildsDictionaries;
    
    /**
     * Rules of json conversion
     * 
     * @var array 
     */
    protected $rules;
    
    /**
     * Enumeration of classes for dynamic creation using UUIDs
     * 
     * @var array 
     */
    protected $uuidables;
    
    /**
     * Handler classes for mapping types
     * 
     * @var array 
     */
    protected $mappingTypeHandlers = [];
    
    /**
     * Cache prefix
     *
     * @var string
     */
    protected $cachePrefix;
    
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->cachePrefix = config('api_core.cache_prefix', 'insegment.api-core.') . 'dto-utils-cache';
        
        $version = $this->getVersionString();
        $appApiPath = rtrim(ChooseVersion::getVersionsPath(), DIRECTORY_SEPARATOR);
        $rulesPath = implode(DIRECTORY_SEPARATOR, [$appApiPath, $version, 'rules.json']);
        $uiidablesPath = implode(DIRECTORY_SEPARATOR, [$appApiPath, $version, 'uuidables.json']);
        $this->rules = $this->getRulesFileContents($rulesPath);
        $this->uuidables = $this->loadUUIDables($uiidablesPath);
        
        foreach ($this->getMappingTypeClasses() as $mappingTypeClass) {
            foreach ($mappingTypeClass::coversTypes() as $type) {
                $this->mappingTypeHandlers[$type] = $mappingTypeClass;
            }
        }
    }

    /**
     * Get global root DataTransferObjects - the DataTransferObjects which and which descendants
     * cover all of the objects that can be used in a transaction alongside any other of them
     * 
     * @return array
     */
    public abstract function getGlobalRootDTOs();
    
    /**
     * Find objects deciding by type of object (import)
     * 
     * @param DataTransferObject $dto
     * @return \Illuminate\Database\Eloquent\Collection|\Illuminate\Database\Eloquent\Model
     * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
     */
    public abstract function findObjectsForImport(DataTransferObject $dto);

    /**
     * Find objects deciding by type of object (export)
     * 
     * @param DataTransferObject $dto
     * @return \Illuminate\Database\Eloquent\Collection
     * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
     */
    public abstract function findObjectsForExport(DataTransferObject $dto);
    
    /**
     * On merge of transaction
     * 
     * This method is to be called before flushing temporary tables into real ones.
     * TransactionSearchScope is still active, so any Eloquent queries will be in union of three tables.
     * 
     * This code is executed within the same MySQL transaction as moving records from temporary to real tables
     * so any exception will force rollback to the state before commit if the post-process step fails
     * 
     * @param \InSegment\ApiCore\Services\Transactor $transcator
     * @return mixed
     */
    public abstract function onTransactionMerge(Transactor $transcator);
    
    /**
     * Post-processing. Any error there will revert to prior-commit.
     * 
     * This method is to be called on a post-process step, temporary tables are already
     * flushed into real ones, but still not dropped, and TransactionSearchScope is already disabled
     * so any Eloquent queries will affect real tables and you can get transacted ids
     * and affect real tables using these IDs.
     * 
     * This code is executed within the same MySQL transaction as moving records from temporary to real tables
     * so any exception will force rollback to the state before commit if the post-process step fails
     * 
     * If onTransactionMerge returned any value, it will be applied as the second argument
     * 
     * @param \InSegment\ApiCore\Services\Transactor $transcator
     * @param mixed $mergeResult
     */
    public abstract function postProcessTransaction(Transactor $transcator, $mergeResult = null);
    
    /**
     * Get result of transaction, if any
     * 
     * @return mixed
     */
    public function getResultOfTransaction()
    {
        return null;
    }
            
    /**
     * Get string representing a version
     * 
     * @return string
     */
    public abstract function getVersionString(): string;
    
    /**
     * Get an instance of DTODefs object, so we can operate on it directly passing array references
     * 
     * (if you use DTODefs::method($array) with reference on array,
     * it will not modify an array, since the method will pass through __call magic)
     * 
     * @return $this
     */
    public function instance()
    {
        return $this;
    }
    
    /**
     * Get known mapping type classes
     * 
     * @return array
     */
    public function getMappingTypeClasses()
    {
        return [
            \InSegment\ApiCore\MappingTypes\Relation::class,
            \InSegment\ApiCore\MappingTypes\Method::class
        ];
    }
    
    /**
     * Get slice structure for slice exchange, by name
     * 
     * @param string $sliceName
     * @return array
     * @throws \Exception
     */
    public function getSliceExchangeStructure(string $sliceName): array
    {
        throw new \Exception("The required slice exchange structure do not exist for name: '{$sliceName}'");
    }
    
    /**
     * Get all rules
     * 
     * @param string|null $class
     * @return array
     */
    public function getRules(string $class = null, $default = null)
    {
        if ($class === null) {
            return $this->rules;
        }
        
        if (!isset($this->rules[$class])) {
            if ($default === null) {
                throw new \Exception("The required rules do not exist for class: '{$class}'");
            } else {
                return $default;
            }
        }
        
        return $this->rules[$class];
    }
    
    /**
     * Get all uuidables
     * 
     * @return array
     */
    public function getUUIDables()
    {
        return $this->uuidables;
    }
    
    /**
     * Whether the class is UUIDable root
     * 
     * @param string $table
     * @return bool
     */
    public function isUUIDable(string $table)
    {
        return isset($this->uuidables[$table]);
    }
    
    /**
     * Check limits for enabling MEMORY engine for specific tables
     * 
     * @return array
     */
    public function getTableMemoryLimits()
    {
        return [
            '*' => 2000,
            (new LogReceivedRecord)->getTable() => 20000,
            (new DeletedTransactionRecord)->getTable() => 20000,
            (new UUIDGeneration)->getTable() => 190000,
        ];
    }
    
    /**
     * Should return:
     * 
     * DTOCharacteristics::DATA_GIVEN_ALL:
     * - Then given data contains at least all unique keys, and the model does not exists
     * - The given data is more than just one unique key value, and the model exists
     * - The given data is not only a unique key value and there are no unique keys
     * 
     * DTOCharacteristics::DATA_GIVEN_KEY:
     * - The given data is only one unique key value, and the model exists
     * 
     * DTOCharacteristics::DATA_GIVEN_NOT_ENOUGH:
     * - The data is not given, the rules are not defined, or rules is key when data is not and vice versa
     * - The given data is an unique key value, and the model does not exists
     * - There is an unique key which is not present in data, and the model does not exists
     * - There are no unique keys and the data has nothing meaningful
     * 
     * Specific cases:
     * - When the auto-increment primary key is present, it is counts as such, but not counts as missing when absent
     * 
     * @param string $class
     * @param mixed $input
     * @param bool $isExists
     * @return int
     */
    public function characterizeGivenData($class, &$input, bool $isExists)
    {
        $rules = &$this->rules[$class] ?? null;
        $rulesIsKey = !is_array($rules);
        
        if (!isset($input) || !isset($rules) || $rulesIsKey !== !is_array($input)) {
            return DTOCharacteristics::DATA_GIVEN_NOT_ENOUGH;
        }
        
        $example = new $class;
        $table = $example->getTable();
        $keyName = $example->getKeyName();
        $isIncrementing = $example->_incrementing ?? $example->incrementing;
        $parser = ParseCreateTable::getInstance();
        $uniqueColumns = $parser->getColumnUniqueData($table);
        $oneOrModeUnique = false;
        $onlyOneUnique = false;
        $notOnlyKey = false;
        $mergeUniqueAttrs = [];
        
        if ($rulesIsKey) {
            $dataKeys = $dataRulesAttributes = [$rules];
        } else {
            $dataKeys = array_keys($input);
            $dataRulesAttributes = [];
            foreach ($dataKeys as $key) {
                if (isset($rules[$key])) {
                    if (is_string($rules[$key])) {
                        $dataRulesAttributes[] = $rules[$key];
                    } else {
                        $notOnlyKey = true;
                    }
                }
            }
        }
        
        foreach ($uniqueColumns as $column) {
            if ($dataRulesAttributes === $column) {
                // all the data is just one unique key
                $onlyOneUnique = true;
                $oneOrModeUnique = true;
                $mergeUniqueAttrs = $column;
                break;
            } else if (array_diff($column, $dataRulesAttributes)) {
                // some unique key is not present
                if (!$isExists && !($isIncrementing && $column === [$keyName])) {
                    return DTOCharacteristics::DATA_GIVEN_NOT_ENOUGH;
                }
            } else {
                $oneOrModeUnique = true;
            }
        }
        
        return ($notOnlyKey || ($oneOrModeUnique && !$onlyOneUnique))
            ? DTOCharacteristics::DATA_GIVEN_ALL
            : ($onlyOneUnique
                ? ($isExists
                    ? DTOCharacteristics::DATA_GIVEN_KEY
                    : DTOCharacteristics::DATA_GIVEN_KEY_NO_CONTENT
                )
                : DTOCharacteristics::DATA_GIVEN_NOT_ENOUGH
            );
    }
    
    /**
     * Carefully get proper related item from $parent
     * 
     * If $noSearch is true, no query will be performed and the relation
     * will be set on the parent with a brand new Model (event if there was something)
     * 
     * @param \Illuminate\Database\Eloquent\Model $parent
     * @param string $relation
     * @param int $searchMode
     * @return Collection
     * @throws \Exception
     */
    public function getOneRelated($parent, string $relation, int $searchMode = DataTransferObject::SEARCH_MODE_NORMAL)
    {
        $relationBuilder = $parent->$relation();
        
        if ($parent->exists && $searchMode !== DataTransferObject::SEARCH_MODE_NEW) {
            if (!$parent->relationLoaded($relation)) {
                $related = $relationBuilder->first();
            } else {
                $related = $parent->getRelation($relation);
            }
            
            if (isset($related) && $searchMode == DataTransferObject::SEARCH_MODE_FRESH) {
                $related->delete();
                unset($related);
            }
        }
        
        if (!isset($related)) {
            return $relationBuilder->newModelInstance();
        }
        
        if ($related instanceof Collection) {
            throw new \Exception("A relation '{$relation}' was hinted to be 'one', but produced Collection instead of value");
        }
        
        return $related;
    }
    
    /**
     * We have an array of objects in json with id's, we need to fetch
     * all of them we have already from the base and match them to keys of array in json by external keys
     * 
     * @param \Illuminate\Database\Eloquent\Builder $builder
     * @param string $externalKey
     * @param array $keys
     * @param int $count
     * @param \Illuminate\Database\Eloquent\Model $template
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getManyWithKeys($builder, string $externalKey, array $keys, int $count, $template)
    {
        /* @see EloquentServiceProvider */
        return $builder->whereIn($externalKey, array_values($keys))->get()->extendToAmount($count, $template);
    }
    
    /**
     * We have an array of objects in json and need toquery the parent Model relation if 
     * it was not loaded, or reuse existing load of Models to get a collection. If the collection has
     * insufficient amount of items, we need to add them
     * 
     * If $noSearch is true, no queries will be done, just new Models created
     * 
     * @param \Illuminate\Database\Eloquent\Model $parent
     * @param string $relation
     * @param int $count
     * @param int $searchMode
     */
    public function getManyRelated($parent, string $relation, int $count, int $searchMode = DataTransferObject::SEARCH_MODE_NORMAL)
    {
        $relationBuilder = $parent->$relation();
        $load = $parent->exists && $searchMode !== DataTransferObject::SEARCH_MODE_NEW;
        $template = $relationBuilder->newModelInstance();
        
        if ($load) {
            if (!$parent->relationLoaded($relation)) {
                $ret = $relationBuilder->get()->extendToAmount($count, $template);
            } else {
                $related = $parent->getRelation($relation);

                if (!$related instanceof Collection) {
                    throw new \Exception("A relation '{$relation}' was hinted to be 'many', but produced value instead of Collection");
                }
                
                $ret = $related->extendToAmount($count, $template);
            }
            
            if ($searchMode === DataTransferObject::SEARCH_MODE_FRESH) {
                foreach ($ret as $key => $model) {
                    if ($model->exists) {
                        $ret[$key] = clone $template;
                        $model->delete();
                    }
                }
            }
        } else {
            $ret = $template->newCollection([$template])->extendToAmount($count, $template);
        }
        
        if ($ret->count() > $count) {
            return $ret->take($count);
        }

        return $ret;
    }
    
    /**
     * We have an array of objects in json with id's, we need to query the parent Model relation if 
     * it was not loaded, or reuse existing load of Models to get a collection of corresponding Models by relation,
     * adding new when needed, then match the collection Models to the keys of array in json by external keys
     * 
     * If $relInfo['noSearch'] equals to true, no queries will be done, just new Models created
     * 
     * @param \Illuminate\Database\Eloquent\Model $parent
     * @param array $relInfo
     * @param int $count
     * @param array $attributes
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getManyRelatedWithKeys($parent, array $relInfo, int $count, array $attributes = [])
    {
        $relationBuilder = $parent->{$relInfo['relation']}();
        $searchMode = Arr::get($relInfo, 'searchMode', DataTransferObject::SEARCH_MODE_NORMAL);
        $load = $parent->exists && $searchMode !== DataTransferObject::SEARCH_MODE_NEW;
        $template = $relationBuilder->newModelInstance($attributes);
        
        if ($load) {
            if (!$parent->relationLoaded($relInfo['relation'])) {
                $ret = $this->getManyWithKeys($relationBuilder, $relInfo['externalKey'], $relInfo['keys'], $count, $template);
            } else {
                $related = $parent->getRelation($relInfo['relation']);

                if (!$related instanceof Collection) {
                    throw new \Exception("A relation '{$relInfo['relation']}' was hinted to be 'many', but produced value instead of Collection");
                }
                
                $ret = $related->whereIn($relInfo['externalKey'], array_values($relInfo['keys']))->extendToAmount($count, $template);
            }
            
            if ($searchMode === DataTransferObject::SEARCH_MODE_FRESH) {
                foreach ($ret as $key => $model) {
                    if ($model->exists) {
                        $ret[$key] = clone $template;
                        $model->delete();
                    }
                }
            }
        } else {
            $ret = $template->newCollection([$template])->extendToAmount($count, $template);
        }
        
        if ($ret->count() > $count) {
            return $ret->take($count);
        }

        return $ret;
    }
    
    /**
     * Produce an array of attributes for the $model, filling all the unique key containing $uniqueKeyPart field
     * 
     * @param array $suppliesForUniques
     * @param string $relatedClass
     * @param array $uniqueKeyParts
     * @return array
     * @throws \Exception
     */
    public function uniquesToAttributes(array $suppliesForUniques, string $relatedClass, array $uniqueKeyParts): array
    {
        $uniques = Transactor::getUniquesForColumns($relatedClass, array_keys($uniqueKeyParts));
        if (!$uniques) {
            $implodeParts = implode(', ', $uniqueKeyParts);
            throw new \Exception("The external key ({$implodeParts}) specified for {$relatedClass} is not a primary or unique key!");
        }

        $attributes = [];
        foreach ($uniques as $fields) {
            foreach ($fields as $field) {
                if (array_key_exists($field, $attributes)) {
                    continue;
                }

                if (array_key_exists($field, $uniqueKeyParts)) {
                    $attributes[$field] = $uniqueKeyParts[$field];
                } else if (array_key_exists($field, $suppliesForUniques)) {
                    $attributes[$field] = $suppliesForUniques[$field];
                } else {
                    $implodeParts = implode(', ', $uniqueKeyParts);
                    throw new \Exception("The external key ({$implodeParts}) specified for {$relatedClass} is a part of unique key, but we do not know another part of it: {$field}");
                }
            }
        }
        
        return $attributes;
    }
    
    /**
     * Get checkable attributes to search for a model
     * 
     * @param string $class
     * @param mixed $data
     * @param array|null $predefined
     * @return array
     */
    public function &collectCheckableAttributes(string $class, &$data, array &$predefined = []): array
    {
        $rules = &$this->rules[$class] ?? null;
        $checkable = [];
        
        if (!is_array($data)) {
            // If the model has external key, we can still search for it using a value from JSON
            if (is_string($rules)) {
                $checkable[$rules] = $data;
            } else if (isset($rules['id'])) {
                $checkable[$rules['id']] = $data;
            }
            
            return $checkable;
        } else if (is_string($rules) && isset($data['id'])) {
            $checkable[$rules] = $data['id'];
        } else {
            // now sure both are arrays
            foreach ($rules as $dotFormat => &$mapping) {
                // filter only plain keys we can check
                if (!is_string($mapping)) {
                    continue;
                }

                // special logic to add a checkable if array_key_exists even if set to null
                $array = $data;
                $segments = explode('.', $dotFormat);
                $last = last($segments);
                foreach ($segments as $segment) {
                    if (is_array($array) && array_key_exists($segment, $array)) {
                        if ($segment === $last) {
                            $checkable[$mapping] = $array[$segment];
                        } else {
                            $array = $array[$segment];
                        }
                    } else {
                        break;
                    }
                }
            }
        }
        
        // predefined attributes will replace everything else and force check of them
        foreach ($predefined as $attribute => &$value) {
            $checkable[$attribute] = $value;
        }
        
        return $checkable;
    }

    /**
     * Perform one or several types of inspections
     * 
     * @param string $rootClass
     * @param array|string $types
     * @param array $results
     * @return array
     */
    public function performInspection(string $rootClass, $types, array &$workLoad = null)
    {
        $results = [];
        with($analyser = new StaticRuleAnalysis($this->rules, $rootClass))
            ->setMethods(
                array_map(function ($type) use ($analyser, $rootClass, &$workLoad, &$results, &$types) {
                    if (isset($workLoad[$type])) {
                        $output = &$workLoad[$type];
                    } else {
                        $output = [];
                    }

                    switch($type) {
                        case 'preload':
                            $results[$type] = &$output;
                            return new PreloadInspection($analyser, $output);
                        case 'relations':
                            $results[$type] = &$output;
                            return StaticRuleAnalysis::relationTypesInspection($output);
                        case 'eagers':
                            $results[$type] = &$output;
                            return StaticRuleAnalysis::eagerLoadsInspection($output);
                        case 'auto-count':
                            $inspectData = [];
                            $results[$type] = &$inspectData;
                            return StaticRuleAnalysis::autoCountInspection($output, $inspectData);
                        case 'enabled-write':
                            $rootExample = new $rootClass;
                            $results[$type] = &$output;
                            $results[$type]['deletable'] = $results[$type]['deletable'] ?? [];
                            $results[$type]['enabled'][$rootClass] = $rootExample->getTable();
                            $results[$type]['key-name'][$results[$type]['enabled'][$rootClass]] = $rootExample->getKeyName();
                            return StaticRuleAnalysis::enabledToWriteInspection($output, Arr::get($workLoad, 'ew-with-read-only', false));
                    }
                }, Arr::wrap($types))
            )->performAnalysis();
            
        return is_array($types) ? $results : $results[$types];
    }
    
    /**
     * Get class of mapping handler or null if no such
     * 
     * @param array $mapping
     * @param bool $onlyCountable
     * @return string|null
     */
    public function classifyMapping($mapping, bool $onlyCountable = false)
    {
        return $this->classifyMapperType(Arr::get($mapping, 'type'), $onlyCountable);
    }
    
    /**
     * Get class of mapping type handler or null if no such
     * 
     * @param string|null $type
     * @param bool $onlyCountable
     * @return string|null
     */
    public function classifyMapperType(string $type = null, bool $onlyCountable = false)
    {
        if ($type && isset($this->mappingTypeHandlers[$type])) {
            if (!$onlyCountable || in_array($type, $this->countableMappingTypes)) {
                return $this->mappingTypeHandlers[$type];
            } else {
                return null;
            }
        }
        
        throw new \Exception("Wrong type in rule mapping: '{$type}'");
    }
    
    /**
     * Get an array of external keys from input and an array of input fields leading to input with these keys
     * 
     * @param string $class
     * @param mixed $input
     * @param string|null $keyName
     * @return array
     */
    public function mapKeysByInput(string $class, $input, string $keyName = null): array
    {
        // TODO: not only take external key into account, but also uniques (do it after versioning)
        $keys = [];
        $inputKeyByExtKeys = [];

        foreach ($input as $inputKey => $inputValue) {
            $keys[] = $key = $this->getExtKeyByData($class, $inputValue, $keyName);
            $inputKeyByExtKeys[$key] = $inputKey;
        }
        
        return ['keys' => $keys, 'inputKeyByExtKeys' => $inputKeyByExtKeys];
    }
    
    /**
     * Change conversion by baseRules if necessary
     * 
     * @param mixed $conversion
     * @param mixed $baseRules
     * @return mixed
     */
    public function rulesByConversion($conversion, $baseRules)
    {
        if (is_array($conversion)) {
            if (array_values($conversion) == $conversion) {
                $ruleSet = $conversion;
                $conversion = [];
                foreach ($ruleSet as $dotFormat) {
                    if (isset($baseRules[$dotFormat])) {
                        $conversion[$dotFormat] = $baseRules[$dotFormat];
                    }
                }
            }
        } else {
            $conversion = $baseRules;
        }
        
        return $conversion;
    }
    
    /**
     * Load rules file
     * 
     * @param string $rulesPath
     * @return array
     */
    protected function getRulesFileContents(string $rulesPath)
    {
        return Cache::rememberForever("{$this->cachePrefix}.rules.{$rulesPath}", function () use ($rulesPath) {
            return json_decode(file_get_contents($rulesPath), true);
        });
    }
    
    /**
     * Load UUIDables
     * 
     * @param string $uiidablesPath
     * @return array
     */
    protected function loadUUIDables(string $uiidablesPath)
    {
        return Cache::rememberForever("{$this->cachePrefix}.uuidables.{$uiidablesPath}", function () use ($uiidablesPath) {
            $fromJSON = json_decode(file_get_contents($uiidablesPath), true);
            $sortedMap = [];

            foreach ($fromJSON as $item) {
                $sortedMap[$item['table']] = $item['dependents'];
            }

            return $sortedMap;
        });
    }
}
