<?php

namespace InSegment\ApiCore\Api;

use InSegment\ApiCore\Facades\DTODefs;
use InSegment\ApiCore\Services\Transactor;
use InSegment\ApiCore\Monads\Maybe;
use InSegment\ApiCore\Api\TransferEngine;
use InSegment\ApiCore\Interfaces\DTOCharacteristics;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Str;
use Illuminate\Support\Arr;

/**
 * Class DataTransferObject represents a root object with the set of sub-objects in an intermediate format
 * written to be a way of transfer some object and all of its descendants between some locations.
 * 
 * Currently it provides basic methods to check type parameter, store/retrieve to/from session, perform local DB search
 * for existing corresponding local entries for the root objects, and allocating necessary amounts of Models for them.
 * 
 * It currently suits only needs of DTODefs, but is extendable. This class should contain methods which will not
 * probably change much in an incompatible ways, so the versions of API may be written as a layer above this class.
 * 
 * After a root object search/allocation is done, DataTransferObject delegates control to the TransferEngine class,
 * which operates on rules versions to receive and write root object descendants into the local DB.
 * 
 * Methods of this class are designed to be used in functional style chains, so it will
 * be possible to write shorter the definitions of versions
 */
class DataTransferObject implements DTOCharacteristics
{
    const DTO_IMPORT_SUCCESS = 'success';
    const DTO_IMPORT_REQUIRE = 'require';
    
    const SEARCH_MODE_NORMAL = 0;
    const SEARCH_MODE_NEW = 1;
    const SEARCH_MODE_FRESH = 2;
    
    const CONTEXT_IMPORT = 0;
    const CONTEXT_EXPORT = 1;
    
    /**
     * Type of object from the route parameter
     * 
     * @var string
     */
    private $type;
    
    /**
     * Input array
     *
     * @var array
     */
    private $input;
    
    /**
     * Reference to the address in the input
     * 
     * @var mixed 
     */
    private $importInput;
    
    /**
     * Result for conditional type checking in functional style
     * (Maybe monad)
     * 
     * @var \App\Monads\Maybe
     */
    private $result;
    
    /**
     * Class of the object to import
     *
     * @var string|null 
     */
    private $class;
    
    /**
     * A builder instance. If the search issues a Builder query, it will be stored until it returns the result
     * When a Builder returns a result, this property is cleared
     * 
     * @var \Illuminate\Database\Eloquent\Builder|null
     */
    private $builder;
    
    /**
     * Context attributes of the retrieved objects. Are set both to $builder->where(...) and to models attributes
     *
     * @var array
     */
    private $context;
    
    /**
     * External key of the object
     * 
     * @var string|null
     */
    private $externalKey;
    
    /**
     * Corelation between an array key from input and an external key of object search result
     *
     * @var array|null
     */
    private $inputKeyByExtKeys;
    
    /**
     * An instance of version of class DTODefs
     * 
     * @var \App\Api\*\DTODefs
     */
    private $dtoDefs;
    
    /**
     * Event to call after processed
     *
     * @var \Closure[]
     */
    private $afterProcess;
    
    /**
     * A field to key export by
     * 
     * @var string|null 
     */
    private $exportByKey;
    
    /**
     * A set of fields to use for export (in dot format)
     * 
     * @var type 
     */
    private $exportFieldSet;
    
    /**
     * Create a DataTransferObject with fresh input
     * 
     * @param string $type
     * @param array $input
     */
    public function __construct($type, &$input, string $importAddress = null)
    {
        $this->dtoDefs = DTODefs::instance();
        $this->type = $type;
        $this->input = &$input;
        if ($importAddress) {
            $this->importAddress($importAddress);
        } else {
            $this->importInput = &$this->input;
        }
        $this->result = Maybe::of(null, $this);
    }
    
    /**
     * One entry and exit point of object import. Processes data it was created with.
     * Delegates the details if result search to the version of class DTODefs
     * 
     * @param bool $update
     * @return mixed
     */
    public function processIncoming(bool $update)
    {
        // clean state
        $this->class = $this->builder = $this->externalKey = $this->inputKeyByExtKeys = null;
        $this->context = $this->afterProcess = [];
        $this->result->set(null);
        
        // find objects
        $result = $this->dtoDefs->findObjectsForImport($this);
        $isCollection = $result instanceof Collection;
        Transactor::fulfilReferenceWildcards($this->class, $this->importInput);
        
        // check data unique keys
        $dataCharacteristic = $isCollection
            ? self::DATA_GIVEN_ALL
            : $this->dtoDefs->characterizeGivenData($this->class, $this->importInput, $result->exists);
        
        switch($dataCharacteristic) {
            case self::DATA_GIVEN_NOT_ENOUGH: return self::DTO_IMPORT_REQUIRE;
            case self::DATA_GIVEN_KEY_NO_CONTENT: return self::DTO_IMPORT_REQUIRE;
            case self::DATA_GIVEN_ALL:
                // create importer
                $importer = new TransferEngine($this->importInput, $this->dtoDefs->getRules(), $update);

                // feed importer with objects
                if ($isCollection && isset($this->externalKey)) {
                    foreach ($this->context as $key => &$value) {
                        data_set($this->importInput, "*.{$key}", $value, false);
                    }

                    $importer->importManyWithKeys($result, $this->externalKey, $this->inputKeyByExtKeys);
                } else {
                    foreach ($this->context as $key => &$value) {
                        data_set($this->importInput, $key, $value, false);
                    }

                    $importer->import($result);
                }
                $importer->endTrace();
            case self::DATA_GIVEN_KEY:
                // call after-process events
                $this->callAfterProcess();
                return self::DTO_IMPORT_SUCCESS;
            default: throw new \Exception("Invalid data characteristics");
        }
    }
    
    /**
     * TODO: make TransferExporter instead of JSONConverts and move conversion rules from models to versions
     * 
     * One entry and exit point of object import. Processes data it was created with.
     * Delegates the details if result search to the JSONConverts trait
     * 
     * @param bool $update
     * @return mixed
     */
    public function processOutgoing()
    {
        // clean state
        $this->class = $this->builder = $this->exportByKey = null;
        $this->context = $this->afterProcess = [];
        $this->result->set(null);
        
        // find objects
        $collection = $this->dtoDefs->findObjectsForExport($this);
        $output = [];
        
        // export models
        $exporter = (new TransferEngine($output, $this->dtoDefs->getRules(), false));
        $exporter->export($collection, $this->exportFieldSet);
        $exporter->endTrace();
        
        if (isset($this->exportByKey)) {
            return collect($output)->keyBy($this->exportByKey)->all();
        }
        
        return $output;
    }
    
    /**
     * Set address of the input which will be imported
     * 
     * @param string $dotFormat
     * @return $this
     */
    public function importAddress(string $dotFormat)
    {
        $importInput = &$this->input;
        
        if ($dotFormat) {
            foreach (explode('.', $dotFormat) as $part) {
                $importInput = &$importInput[$part];
            }
        }
        
        $this->importInput = &$importInput;
        return $this;
    }
    
    /**
     * Set class to use for operations
     * 
     * @param string $class
     * @return $this
     */
    public function setClass($class)
    {
        $this->class = $class;
        
        return $this;
    }
    
    /**
     * Set context attributes. This adds attributes both to the where condition and to the attributes of results
     * 
     * @param array $attributes
     * @return $this
     */
    public function setContext($attributes)
    {
        $this->context = $attributes;

        with($this->builder = $this->builder ?: $this->class::query())->where($attributes);
        
        return $this;
    }
    
    /**
     * Count data by its address in input
     * 
     * @param string $dotFormat
     * @return int
     */
    public function countData($dotFormat = '')
    {
        $toCount = &$this->importInput;
        
        if ($dotFormat) {
            foreach (explode('.', $dotFormat) as $part) {
                $toCount = &$toCount[$part];
            }
        }
        
        return count($toCount);
    }
    
    /**
     * Field to key by when exporting
     * 
     * @param string $field
     * @return $this
     */
    public function setKeyBy(string $field)
    {
        $this->exportByKey = $field;
        
        return $this;
    }
    
    /**
     * Field to key by when exporting
     * 
     * @param array $fieldSet
     * @return $this
     */
    public function setFieldSet(array $fieldSet)
    {
        $this->exportFieldSet = $fieldSet;
        
        return $this;
    }
    
    /**
     * A bit complicated task: uses both previously computed constraints from $this->builder, if any,
     * and external keys from $this->data, to find objects both matching custom conditions and having
     * external keys set to specified in $this->data, or creates new objects with such external keys
     * in amount of supplied $this->data objects
     * 
     * @param array $attributes
     * @param string|null $externalKey
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function takeOrNewExtKeys($attributes = [], $externalKey = null)
    {
        if (!empty($this->context)) {
            $attributes = $this->context + $attributes;
        }
        
        $this->inputKeyByExtKeys = null;
        
        if (isset($this->builder)) {
            $builder = $this->builder;
            $this->builder = null;
        } else {
            $builder = $this->class::query();
        }

        if (($this->externalKey = $externalKey ?? $this->dtoDefs->getExternalKeyDefaultName($this->class))) {
            $mapKeys = $this->dtoDefs->mapKeysByInput($this->class, $this->importInput, $externalKey);
            $this->inputKeyByExtKeys = &$mapKeys['inputKeyByExtKeys'];
            $keys = &$mapKeys['keys'];

            return $this->class::unguarded(function () use ($builder, &$keys, $attributes) {
                /* @see EloquentServiceProvider */
                return $builder->whereIn($this->externalKey, array_values($keys))->takeOrNewAmount($this->countData(), $attributes);
            });
        } else {
            return $builder->takeOrNewAmount($this->countData(), $attributes);
        }
    }
    
    /**
     * Make new object with attributes
     * 
     * @param array $attributes
     * @return mixed
     */
    public function make($attributes = [])
    {
        return $this->class::unguarded(function () use ($attributes) {
            return with(new $this->class)->fill($attributes);
        });
    }
    
    /**
     * Find the first object or make new with attributes
     * 
     * @param array $attributes
     * @return mixed
     */
    public function firstOrMake($attributes = [])
    {
        return $this->class::unguarded(function () use ($attributes) {
            return $this->class::firstOrNew($attributes)->fill($attributes);
        });
    }
    
    /**
     * Set the object, stored as a result of previous computation, to the Transactor session
     * 
     * @param mixed $object
     * @return $this
     */
    public function setSession($object)
    {
        Transactor::setSessionObject($this->class,
            ($object instanceof Model
                ? $object->getKey()
                : ($object instanceof Collection
                    ? $object->pluck(with(new $this->class)->getKeyName())
                    : $object
                ))
        );
        
        return $this;
    }
    
    /**
     * This method computes conditions and if they match and there are some output from them, gives it
     * to the closure with $this rewritten to the instance of DataTransferObject
     * 
     * @param array $options
     * @param \Closure $closure
     * @param \Closure|null $orMaybe
     * @return \App\Monads\Maybe
     */
    public function check(array $options, \Closure $closure, \Closure $orMaybe = null): Maybe
    {
        $arguments = [];
        
        return Maybe::of(
            (isset($options['data'])
                ? $this->withData($options['data'], $arguments)
                : true
            
            ) && (isset($options['session'])
                ? $this->withSession($options['session'], $arguments)
                : true
            
            ) && (isset($options['dataType'])
                ? $this->withDataType($options['dataType'])
                : true
            
            ) ?: null, $this)
            ->map(function (bool $true) use ($closure, &$arguments) {
                return call_user_func_array($closure->bindTo($this, static::class), $arguments);
            });
    }
    
    /**
     * Add event to when process is done
     * 
     * @param \Closure $closure
     * @return $this
     */
    public function afterProcess(\Closure $closure)
    {
        $this->afterProcess[] = $closure->bindTo($this, static::class);
        
        return $this;
    }
    
    /**
     * Load eagers with the query
     * 
     * @param string $type
     * @param string|null $prefix
     * @param array|null $eagers
     * @return $this
     */
    public function eagers(string $type, string $prefix = null, array $eagers = null)
    {
        if (!isset($this->builder)) {
            $this->builder = $this->class::query();
        }
        
        $allEnabled = Transactor::getEnabledEagers($this->class, ['all', $type]);
        
        if (isset($eagers)) {
            $filtered = array_filter($allEnabled, function ($item) use (&$eagers) {
                return Str::startsWith($item, $eagers);
            });
        } else if ($prefix) {
            $filtered = array_filter($allEnabled, function ($item) use (&$prefix) {
                return Str::startsWith($item, $prefix);
            });
        } else {
            $filtered = $allEnabled;
        }
        
        $realEagers = $prefix
            ? array_map(
                function ($item) use (&$prefix) {
                    return last(explode($prefix, $item));
                }, $filtered
            )
            : $filtered;

                
        if ($realEagers) {
            $this->builder->with($realEagers);
        }
        
        return $this;
    }
    
    /**
     * Add scopes to builder
     * 
     * @param array $scopes
     * @return $this
     */
    public function scopes(array $scopes)
    {
        if (!isset($this->builder)) {
            $this->builder = $this->class::query();
        }
        
        foreach($scopes as $scope) {
            $this->builder->$scope();
        }
        
        return $this;
    }
    
    /**
     * Call after-process events
     * 
     * @return $this
     */
    private function callAfterProcess()
    {
        foreach ($this->afterProcess as $closure) {
            $closure();
        }
        
        return $this;
    }
    
    /**
     * Check some value by key in supplied $this->data, or return false, if none
     * 
     * @param array $dotFormats
     * @param array $arguments
     * @return boolean
     */
    private function withData($dotFormats, &$arguments)
    {
        foreach ($dotFormats as $key => $value) {
            if (!is_int($key)) {
                $dotFormat = $key;
                $default = $value;
                $optional = true;
            } else {
                $dotFormat = $value;
                $default = null;
                ($optional = substr($dotFormat, -1) === '?') && ($dotFormat = substr($dotFormat, 0, -1));
            }
            
            if (!is_array($this->input) || !Arr::has($this->input, $dotFormat)) {
                if ($optional) {
                    $arguments[] = $default;
                    continue;
                } else {
                    return false;
                }
            }
            
            $toPush = &$this->input;
            
            if ($dotFormat) {
                foreach (explode('.', $dotFormat) as $part) {
                    $toPush = &$toPush[$part];
                }
            }
            
            $arguments[] = $toPush;
        }
        return true;
    }
    
    /**
     * Give the required objects stored in the Transactor session to the $arguments
     * if they are present, or return false if not
     * 
     * @param array $sessionRequirements
     * @param array $arguments
     * @return boolean
     */
    private function withSession($sessionRequirements, &$arguments)
    {
        foreach ($sessionRequirements as $class) {
            $object = Transactor::getSessionObject($class);
            if (!isset($object)) {
                return false;
            }
            
            $arguments[] = $object;
        }
        
        return true;
    }
    
    /**
     * Check type of $this->data
     * 
     * @param string $dataType
     * @return bool
     */
    private function withDataType($dataType)
    {
        return ($dataType == 'array' && is_array($this->input))
            || ($dataType == 'string' && is_string($this->input))
            || ($dataType == 'int' && (is_int($this->input) || ctype_digit($this->input)))
            || ($dataType == 'float' && is_float($this->input));
    }
    
    /**
     * Magic __call method.
     * 
     * Methods resultSomething(...) are translated into Maybe monad of result: $this->result->something(...)
     * with any closures in arguments to Maybe monad binded $this to the instance of DataTransferObject
     * 
     * Methods querySomething(...) are translated into query builder: $builder->something(...)
     * 
     * Methods ifTypeSomething check for $this->type == something, and allows to build DTODefs definitions
     * of search for object in functional style rewriting $this of the closures to the instance of DataTransferObject
     * 
     * @param string $method
     * @param array $arguments
     * @return mixed
     * @throws BadMethodCallException
     */
    public function __call($method, $arguments) {
        if (strpos($method, 'result') === 0) {
            $maybeMethod = lcfirst(substr($method, 6));
            $return = $this->result->$maybeMethod(...$arguments);
            
            return $return === $this->result ? $this : $return;
        } else if (strpos($method, 'query') === 0 && isset($this->class)) {
            $builderMethod = lcfirst(substr($method, 5));
            
            if (!isset($this->builder)) {
                $this->builder = $this->class::query();
            }
            
            $this->class::unguard();
            $result = $this->builder->$builderMethod(...$arguments);
            $this->class::reguard();
            
            if ($result instanceof Builder) {
                return $this;
            } else {
                $this->builder = null;
                return $result;
            }
        } else if (strpos($method, 'ifType') === 0) {
            if (Str::camel($this->type) === lcfirst(substr($method, 6))) {
                $closure = $arguments[0];
                unset($arguments[0]);
                return $closure->call($this, ...$arguments);
            }
            
            return false;
        }
        
        throw new \BadMethodCallException("Method {$method} does not exist.");
    }
}
