<?php

namespace InSegment\ApiCore\MappingTypes;

use Illuminate\Support\Arr;
use InSegment\ApiCore\Api\TransferEngine;
use InSegment\ApiCore\Inspections\AutoCountInspection;

use Closure;

/**
 * A template class for building custom mapping types
 */
abstract class MappingType
{
    /**
     * Action to be done on import
     * 
     * @param \InSegment\ApiCore\Api\TransferEngine $this
     * @param array $mapping
     * @return mixed
     * @var \Closure
     */
    private $onImport;
    
    /**
     * Action to be done on export
     * 
     * @param \InSegment\ApiCore\Api\TransferEngine $this
     * @param array $mapping
     * @return mixed
     * @var \Closure
     */
    private $onExport;
    
    /**
     * Non-public constructor (to be owerriden by public one)
     * Binds import and export closures to TransferEngine instance
     * 
     * @param TransferEngine $engine
     * @param Closure $onImport
     * @param Closure $onExport
     */
    protected function __construct(TransferEngine $engine, Closure $onImport, Closure $onExport) {
        $this->onImport = $onImport->bindTo($engine, TransferEngine::class);
        $this->onExport = $onExport->bindTo($engine, TransferEngine::class);
    }
    
    /**
     * Get result of using $this->onImport with TransferEngine and $mapping
     * 
     * @param array $mapping
     */
    public function import(&$mapping)
    {
        return ($this->onImport)($mapping);
    }
    
    /**
     * Get result of using $this->onExport with TransferEngine and $mapping
     * 
     * @param array $mapping
     */
    public function export(&$mapping)
    {
        return ($this->onExport)($mapping);
    }
    
    /**
     * Get information about external key name (and, optionally, value)
     * 
     * @param \InSegment\ApiCore\Api\V_1_0\DTOUtil $dtoDefs
     * @param array $mapping
     * @param array $input
     * @param array $param
     * @return array
     */
    public static function getExtKeysInfo($dtoDefs, $mapping, $input = null, array $param = []): array
    {
        $relatedClass = static::getRelatedClass($mapping);
        
        // for the 'find' type, prefer the specified key for search (maybe null) to default external key name
        $extKey = static::getKeyFromMapping($mapping) ?? $dtoDefs->getExternalKeyDefaultName($relatedClass);
        return $dtoDefs->getExtKeyInfo($relatedClass, $extKey, $input, $param);
    }
    
    /**
     * Get keys needed to be set for a successful search for a record by mapping
     * 
     * @param \InSegment\ApiCore\Facades\DTODefs $dtoDefs
     * @param mixed $mapping
     * @param mixed $baseRules
     * @param string $relatedClass
     * @param array $extKeysInfo
     * @return array
     */
    public static function getSearchAttributeKeys($dtoDefs, $mapping, $baseRules, $relatedClass, $extKeysInfo)
    {
        if (isset($mapping['uniqueSupply'])) {
            $convertedRules = $dtoDefs->rulesByConversion($mapping['uniqueSupply'], $baseRules);

            if (!is_array($convertedRules)) {
                $suppliesForUniques = null;
            } else {
                $suppliesForUniques = [];
                $conversionCount = count($convertedRules);
                foreach ($convertedRules as $dotFormat => $setMapping) {
                    if ($setMapping === null) {
                        continue;
                    }
                    
                    if ($dotFormat === '-' && $conversionCount === 1) {
                        $suppliesForUniques = null;
                        break;
                    }
                    
                    Arr::set($suppliesForUniques, $dotFormat, null);
                }
            }
        } else {
            $suppliesForUniques = [];
        }
        
        return array_keys($dtoDefs->uniquesToAttributes($suppliesForUniques, $relatedClass, $extKeysInfo));
    }
    
    /**
     * Should return an array of strings representing types of mappings
     * which are to be handled by this class
     * 
     * @return array
     */
    public abstract static function coversTypes(): array;
    
    /**
     * Enumerate relation type, return string with full new address if enumeration is to recurse
     * 
     * @param array $output
     * @param string $address
     * @param string $class
     * @param string $dotFormat
     * @param mixed $mapping
     * @return bool
     */
    public static function enumerateRelationType(&$output, $address, $class, $dotFormat, &$mapping): bool
    {
        return static::importsRelation($mapping);
    }
    
    /**
     * Get relation name from mapping
     * 
     * @param string $class
     * @param array $mapping
     * @return null
     */
    public static function getRelationName($class, &$mapping)
    {
        $relationName = Arr::get($mapping, 'relation');
        return $relationName && method_exists(new $class, $relationName) ? $relationName : null;
    }
    
    /**
     * Get related class for relation of mapping type
     * 
     * @param array $mapping
     * @return null
     */
    public static function getRelatedClass(&$mapping)
    {
        return null;
    }
    
    /**
     * Get class name of AutoCountInspection
     * 
     * @return string
     */
    public static function getAutoCountInspectionClass(): string
    {
        return AutoCountInspection::class;
    }
    
    /**
     * Whether this MappingType relation should look for an input to process its object
     * 
     * @param array $mapping
     * @return bool
     */
    public static function importsRelation(&$mapping): bool
    {
        return false;
    }
    
    /**
     * Check whether the mapping type is auto-countable (one-to-one with parent)
     * 
     * @param array $mapping
     * @return bool
     */
    public static function isAutoCount(&$mapping): bool
    {
        return false;
    }
    
    /**
     * Get key from mapping
     * 
     * @param array $mapping
     * @return array|string|null
     */
    protected static function getKeyFromMapping(&$mapping)
    {
        return null;
    }
    
    /**
     * Check whether records from the table of the related model could be deleted
     * 
     * @param array $mapping
     * @return bool
     */
    public static function isDeletionEnabled(&$mapping): bool
    {
        return false;
    }
    
}
