<?php

namespace InSegment\ApiCore\MappingTypes;

use Illuminate\Support\Arr;
use InSegment\ApiCore\Api\TransferEngine;

class Relation extends MappingType
{
    /**
     * Should return an array of strings representing types of mappings
     * which are to be handled by this class
     * 
     * @return array
     */
    public static function coversTypes(): array
    {
        return ['one', 'many', 'find'];
    }

    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Api\TransferEngine $engine
     */
    public function __construct(TransferEngine $engine)
    {
        /**
         * @var \InSegment\ApiCore\Api\TransferEngine $this
         */
        parent::__construct(
            $engine,
                
            /**
             * Import function
             * 
             * @var \InSegment\ApiCore\Api\TransferEngine $this
             */
            function ($mapping) {
                $this->relateToAttribute();
            },
                    
            /**
             * Export function
             * 
             * @var \InSegment\ApiCore\Api\TransferEngine $this
             */
            function ($mapping) {
                $this->exportRelation($mapping);
            }
        );
    }
    
    /**
     * Enumerate relation type, return string with full new address if enumeration is to recurse
     * 
     * @param array $output
     * @param string $address
     * @param string $class
     * @param string $dotFormat
     * @param mixed $mapping
     * @return bool
     */
    public static function enumerateRelationType(&$output, $address, $class, $dotFormat, &$mapping): bool
    {
        $fullAddress = $dotFormat === '-' ? $address : "{$address}.{$dotFormat}";
        $relationTypeName = basename(get_class(with(new $class)->{$mapping['relation']}()));

        $output[$fullAddress] = isset($output[$fullAddress])
            ? "{$output[$fullAddress]} => {$relationTypeName} {$mapping['class']}"
            : "{$relationTypeName} {$mapping['class']}";

        if (isset($mapping['extra'])) {
            $output[$fullAddress] .= ' (' . implode(', ', $mapping['extra']) . ')';
        }
        
        return static::importsRelation($mapping);
    }
    
    /**
     * Get related class for relation of mapping type
     * 
     * @param array $mapping
     * @return string
     */
    public static function getRelatedClass(&$mapping)
    {
        return $mapping['class'];
    }
    
    /**
     * Whether this MappingType relation should look for an input to process its object
     * 
     * @param array $mapping
     * @return bool
     */
    public static function importsRelation(&$mapping): bool
    {
        return $mapping['type'] !== 'find';
    }
    
    /**
     * Check whether the mapping type is auto-countable (one-to-one with parent)
     * 
     * @param array $mapping
     * @return bool
     */
    public static function isAutoCount(&$mapping): bool
    {
        return $mapping['type'] === 'one';
    }
    
    /**
     * Get key from mapping
     * 
     * @param array $mapping
     * @return array|string|null
     */
    protected static function getKeyFromMapping(&$mapping)
    {
        // for the 'find' type, prefer the specified key for search (maybe null) to default external key name
        return array_key_exists('key', $mapping) ? $mapping['key'] : null;
    }
    
    /**
     * Check whether records from the table of the related model could be deleted
     * 
     * @param array $mapping
     * @return bool
     */
    public static function isDeletionEnabled(&$mapping): bool
    {
        return in_array('fresh', Arr::get($mapping, 'extra', []));
    }
}
