<?php

namespace InSegment\ApiCore\Api;

// MAKE BETTER 4: parse DB::raw("FIND_IN_SET('value', (`?table`?)?`?column`?) > 0")
// MAKE BETTER 6: morphTo several classes (usable for Industry)
// MAKE BETTER 7: use 'through' MappingType instead of '-' rule
// MAKE BETTER 8: do not check types directly in transfer engine, delegate it to MappingType (i.e. 'find', 'one' etc)
// MAKE BETTER 9: in transfer importer, when importing in order by relation with many, count-in uniques
// MAKE BETTER 10: support contexts in export, they should be moved up the tree, not copied
use Illuminate\Support\Arr;

class TransferEngine extends RuleAnalysis
{
    use Concerns\ImportsObjects;
    use Concerns\ExportsObjects;
    
    /**
     * Model being imported/exported
     * 
     * @var \Illuminate\Database\Eloquent\Model 
     */
    protected $model;
    
    /**
     * Class of the Model being imported
     *
     * @var string 
     */
    protected $modelClass;
    
    /**
     * Currently processed mapping
     * 
     * @var array|null
     */
    protected $mapping;
    
    /**
     * Current context
     * 
     * @var array|null
     */
    protected $context;
    
    /**
     * Current context type
     * 
     * @var bool|null
     */
    protected $contextType;
    
    /**
     * Decompressed and decoded data
     *
     * @var mixed
     */
    protected $input;
    
    /**
     * Run-time variable of current model's downstream relations
     * 
     * @var array 
     */
    protected $modelDownStream;
    
    /**
     * If the parent Model have relation to current Model, it is then used
     * to affect current Model attributes by constraints
     * 
     * @var \Illuminate\Database\Eloquent\Relations\Relation|null
     */
    protected $relationToModel;
    
    /**
     * "History" of descending by dot-formatted rules
     * 
     * @var array
     */
    protected $stack;
    
    /**
     * Stack of ascended properties
     *
     * @var array 
     */
    protected $ascends = [];
    
    /**
     * Mapping type handlers by type
     * 
     * @var array 
     */
    protected $mappers;
    
    /**
     * Dictionaries and models preloaded ahead in transaction loaded for classes
     * 
     * @var array
     */
    protected $preloads;
    
    /**
     * Omittance of rules
     * 
     * @var array 
     */
    protected $omit;
    
    /**
     * Constructor
     * 
     * @param array $input
     * @param array $rules
     * @param bool $update
     */
    public function __construct(&$input, $rules, bool $update)
    {
        parent::__construct($rules);
        
        $this->stack = [];
        $this->input = &$input;
        $this->update = $update;
        $this->mappers = [];
        $this->preloads = [];
        $this->omit = [];
        
        foreach ($this->dtoDefs->getMappingTypeClasses() as $class) {
            $instance = new $class($this);
            
            foreach ($class::coversTypes() as $type) {
                $this->mappers[$type] = $instance;
            }
        }
    }

    /**
     * Should descend the given array of parameters doing the opposite operation of ascendParams in reverse order
     * Stores array of ascended params
     * 
     * @param array $params
     * @return null
     */
    protected function descendParams(array $params)
    {
        if ($this->trace) {
            $level = count($this->ascends);
            $caller = debug_backtrace()[1]['function'];
            $class = $this->modelClass;
            $extra = [];
            if ($caller === 'mappingToAttribute') {
                $extra['dotFormat'] = $params['dotFormat'] ?? '';
            }
            
            $this->pushLogString('descend', json_encode(['caller' => $caller, 'level' => $level, 'class' => $class, 'extra' => $extra, 'paramKeys' => array_keys($params)]));
        }
        $ascended = [];
        
        if (isset($params['mapping'])) {
            $ascended['mapping'] = $this->descendMapping($params['mapping']);
        }
        
        if (isset($params['model'])) {
            $ascended['model'] = $this->descendModel($params['model']);
        }
        
        if (isset($params['relation'])) {
            $ascended['relation'] = $this->descendRelation($params['relation']);
        }
        
        if (isset($params['dotFormat'])) {
            $ascended['dotFormat'] = $this->descendDotFormat($params['dotFormat']);
        }
        
        if (isset($params['input'])) {
            $ascended['input'] = $this->descendInput($params['input']);
        }
        
        if (isset($params['context'])) {
            $ascended['context'] = $this->descendContext($params['context']);
        }
        
        if (isset($params['omit'])) {
            $ascended['omit'] = $this->descendOmit();
        }
        
        $this->ascends[] = $ascended;
    }
    
    /**
     * Should ascend the given array of parameters doing the opposite operation of descendParams in reverse order
     * Restores values of ascended params
     * 
     * @param array $params
     * @return null
     */
    protected function ascendParams(array $params)
    {
        $ascended = array_pop($this->ascends);
        
        if (isset($params['omit'])) {
            $this->omit = $ascended['omit'];
        }
        
        if (isset($params['context'])) {
            $this->ascendContext($ascended['context']);
        }
        
        if (isset($params['input'])) {
            $this->ascendInput($ascended['input']);
        }
        
        if (isset($params['dotFormat'])) {
            $this->ascendDotFormat($ascended['dotFormat']);
        }
        
        if (isset($params['relation'])) {
            $this->ascendRelation($ascended['relation']);
        }
        
        if (isset($params['model'])) {
            $this->ascendModel($ascended['model']);
        }
        
        if (isset($params['mapping'])) {
            $this->ascendMapping($ascended['mapping']);
        }

        if ($this->trace) {
            $level = count($this->ascends);
            $caller = debug_backtrace()[1]['function'];
            $class = $this->modelClass;
            $extra = [];
            if ($caller === 'mappingToAttribute') {
                $extra['dotFormat'] = $params['dotFormat'] ?? '';
            }
            
            $this->pushLogString('ascend', json_encode(['caller' => $caller, 'level' => $level, 'class' => $class, 'extra' => $extra, 'paramKeys' => array_keys($params)]));
        }
    }
    
    /**
     * Descend model parameter
     * 
     * @param \Illuminate\Database\Eloquent\Model $model
     * @return array
     */
    protected function descendModel($model)
    {
        $ascended = ['model' => $this->model, 'modelClass' => $this->modelClass, 'modelDownStream' => $this->modelDownStream];

        $this->model = $model;
        $this->modelClass = get_class($model);
        $this->modelDownStream = [];
        
        return $ascended;
    }
    
    /**
     * Ascend model parameter
     * 
     * @param array $ascended
     * @return null
     */
    protected function ascendModel(array &$ascended)
    {
        $this->model = &$ascended['model'];
        $this->modelClass = &$ascended['modelClass'];
        $this->modelDownStream = &$ascended['modelDownStream'];
    }

    /**
     * Descend relation parameter
     * 
     * @param \Illuminate\Database\Eloquent\Relations\Relation $relation
     * @return \Illuminate\Database\Eloquent\Relations\Relation
     */
    protected function descendRelation($relation)
    {
        $ascendedRelationToModel = $this->relationToModel;
        $this->relationToModel = $relation;
        return $ascendedRelationToModel;
    }
    
    /**
     * Ascend relation parameter
     * 
     * @param \Illuminate\Database\Eloquent\Relations\Relation $ascendedRelation
     * @return null
     */
    protected function ascendRelation($ascendedRelation)
    {
        $this->relationToModel = $ascendedRelation;
    }
    
    /**
     * Descend mapping parameter
     * 
     * @param mixed $mapping
     * @return mixed
     */
    protected function descendMapping(&$mapping)
    {
        $ascended = ['mapping' => &$this->mapping, 'context' => &$this->context];
        $this->mapping = &$mapping;
        $this->context = &$this->importContextByMapping();
        
        return $ascended;
    }
    
    /**
     * Ascend mapping parameter
     * 
     * @param mixed $ascended
     * @return null
     */
    protected function ascendMapping(&$ascended)
    {
        $this->exportContextByMapping();
        $this->mapping = &$ascended['mapping'];
        $this->context = &$ascended['context'];
    }
    
    /**
     * Descend context type parameter
     * 
     * @param int $contextType
     * @return null
     */
    protected function descendContext($contextType)
    {
        $ascendedContextType = $this->contextType;
        $this->contextType = $contextType;
        
        if ($this->contextType === DataTransferObject::CONTEXT_IMPORT) {
            if (!empty($this->context)) {
                if (!is_array($this->input)) {
                    if (($key = Arr::get($this->mapping, 'key'))) {
                        $this->input = ['id' => $this->input];
                    } else {
                        throw new \Exception('Unable to set context on non-array input with no key in mapping');
                    }
                }

                foreach ($this->context as $dotFormat => &$value) {
                    data_set($this->input, $dotFormat, $value, false);
                }
            }
            
            /*if (!empty($this->mapping['omit'])) {
                foreach ($this->mapping['omit'] as $dotFormat) {
                    Arr::forget($this->input, $dotFormat);
                }
            }*/
        }
        
        return $ascendedContextType;
    }
    
    /**
     * Ascend context type parameter
     * 
     * @param int|null $ascendedContextType
     * @return null
     */
    protected function ascendContext($ascendedContextType)
    {
        if ($this->contextType === DataTransferObject::CONTEXT_EXPORT && is_array($this->input)) {
            if (!empty($this->context)) {
                foreach ($this->context as $dotFormat => &$value) {
                    $value = Arr::get($this->input, $dotFormat, $value);
                    Arr::forget($this->input, $dotFormat);
                }
            }
            
            /*if (!empty($this->mapping['omit'])) {
                foreach ($this->mapping['omit'] as $dotFormat) {
                    Arr::forget($this->input, $dotFormat);
                }
            }*/
        }
        
        $this->contextType = $ascendedContextType;
    }
    
    /**
     * Descend omit parameter
     * 
     * @return array
     */
    protected function descendOmit()
    {
        $ascendedOmit = $this->omit;
        foreach (Arr::get($this->mapping, 'omit', []) as $omit) {
            $this->omit[$omit] = true;
        }
        
        return $ascendedOmit;
    }
    
    /**
     * Descend input parameter
     * 
     * @param mixed $input
     * @return null
     */
    protected function descendInput(&$input)
    {
        $ascended = &$this->input;
        $this->input = &$input;
        return $ascended;
    }
    
    /**
     * Ascend input parameter
     * 
     * @return null
     */
    protected function ascendInput(&$ascended)
    {
        $this->input = &$ascended;
    }

    /**
     * Descend dotFormat parameter
     * 
     * @param string $dotFormat
     * @return array
     */
    protected function descendDotFormat($dotFormat)
    {
        // store the current state of input
        $ascended = ['input' => &$this->input, 'dotFormat' => $dotFormat];
        
        // add dotFormat to descendance stack
        array_push($this->stack, $dotFormat);

        // descend
        if ($dotFormat !== '-') {
            foreach (explode('.', $dotFormat) as $partOfPath) {
                $this->input = &$this->input[$partOfPath];
            }
        }
        
        return $ascended;
    }
    
    /**
     * Ascend dotFormat parameter
     * 
     * @param array $ascendedInput
     * @return null
     */
    protected function ascendDotFormat(&$ascended)
    {
        $this->input = &$ascended['input'];
        
        if (isset($ascended['dotFormat']) && array_pop($this->stack) !== $ascended['dotFormat']) {
            throw new \Exception("Something went wrong! Descendance stack was affected!");
        }
    }

    /**
     * Get new objectContext changed for mapping context and defaults
     * 
     * @return array
     */
    private function &importContextByMapping(): array
    {
        $newContext = Arr::get($this->mapping, 'defaults', []);
        
        foreach (Arr::get($this->mapping, 'context', []) as $key => $value) {
            if (is_int($key)) {
                $contextValue = Arr::get($this->input, $value) ?? Arr::get($this->context, $value);
            } else {
                $contextValue = Arr::get($this->input, $key) ?? Arr::get($this->context, $key);
            }
            
            if (!isset($newContext[$value])) {
                $newContext[$value] = $contextValue;
            }
        }
        
        if (isset($this->mapping['exportContext'])) {
            $export = [];
            
            $this->exportModel(['input' => &$export, 'dotFormat' => '-', 'omit' => true], $this->mapping['exportContext']);
            
            if ($export) {
                $newContext += Arr::dot($export);
            }
        }
        
        return $newContext;
    }
    
    /**
     * Set input based on objectContext for mapping context and defaults
     * 
     * @return null
     */
    private function exportContextByMapping()
    {
        $defaults = $this->mapping['defaults'] ?? [];
        
        foreach (($this->mapping['context'] ?? []) as $key => $value) {
            if (is_array($this->input)) {
                $contextValue = $this->context[$value] ?? $defaults[$value] ?? null;
                
                if (is_int($key)) {
                    data_set($this->input, $value, $contextValue, false);
                } else {
                    data_set($this->input, $key, $contextValue, false);
                }
            }
        }
        
        /*if (isset($this->mapping['exportContext'])) {
            $export = [];
            
            $this->exportModel(['input' => &$export, 'dotFormat' => '-', 'omit' => true], $this->mapping['exportContext']);
            
            if ($export) {
                $newContext += array_dot($export);
            }
        }*/
    }
}
