## Format of Sections

A section's header is the full path of the class on class path to the Model which the set of rules
will be applied to. A section consists of a set of rules for mapping fields of json input to:
- `"field": "direct_field"` - direct mapping from a dot-formatted path of json object to plain fields of a Model
- `"field": {"type": "...", ...}` - complex mapping from a dot-formatted path of json object

## Types of maping
There are several different types of supported complex mappings. The main are relational ones. Then come methods. If that is not enough, you can extend abstract class MappingType and write your own one.

### Mapping 'one' object or value
This type expects the object or a value on dot-formatted json path and will use specified relation method on the parent Eloquent Model, then import it (find or create new) using the rules described it the section named after the class of the target Eloquent Model.
```
{
  "type": "one",
  "class": "App\\\Models\\...",
  "relation": "...",
  ...
}
```

Example of expected values:
```
"parentField": {
  "id":123,
  "field": "value"
}
```
```
"parentField": "value"
```

### Mapping 'many' objects or values

This works the same way as the previous one, the main difference is that is expects a series of objects by dot-formatted json path in an array brackets.
```
{
  "type": "many",
  "class": "App\\\Models\\...",
  "relation": "...",
  ...
}
``` 

Example of accepted values:
```
"parentField": [
   {
     "id":1,
     "value": "abc"
   },
   {
     "id":2,
     "value": "acd"
   }
]
```
```
"parentField": ["value1", "value2"]
```

### Search for existing objects - 'find'

Similar variation to the 'one' type again, but it doesn't import object. It looks for existing one instead. It can only be used with BelongsTo (or MorphTo) relations because it implies that the json object value provided will specify a value of a key of an object.
```
{
  "type": "find",
  "class": "App\\\Models\\...",
  "relation": "...",
  ...
}
```

Accepts value of the related Model key:
```
"parentField": "value"
```

By default, it uses Model's external key, but you may want specific one adding to the mapping:
```
{
  "type": "find",
  ...
  "key": "specific_key",
  ...
}
```

To be able to auto-load the records massively on import, you may want to specify:

```
{
  "type": "find",
  ...
  "referenceWidcard": "address.from.root.*.id",
  ...
}
```

It will then load Models, which key or specific key corresponds to all the values found in json using address from root of the object

You can also specify multi-key search like this:

```
{
  "type": "find",
  ...
  "key": {
    "some_id": "some_id_in_json",
    "other_id": "other_id_in_json"
  },
  "referenceWidcard": {
    "some_id": "address.from.root.*.id_in_json",
    "other_id": "address.from.root.*.other_id_in_json",
  }
  ...
}
```

### Using 'method' for import and for export of value

Specify "methodTo" to use on the Model when importing and invoke it with a value or an object from json. Specify "methodFrom" to use on the Model when exporting and set the field of json by the method's returned result. Limitations of this mapping type is that is is cannot be statically analysed for auto-counting, and several other cases.

```
{
  "type": "method",
  "methodTo": "methodOnTheModel",
  "methodFrom": "methodOnTheModel",
  ...
}
```

You can also specify static methods of some classes (not necessary different):
```
{
  "type": "method",
  "methodTo": ["App\\SomeClass", "methodOnTheClass"]`,
  "methodFrom": ["App\\SomeOtherClass", "methodOnTheOtherClass"]`,
  ...
}
```


## Contexts

There is away to pass properties from the objects up the tree to their descendants. They will be threated in descendants as if they were in input for the object unless the input for the object actually contains that field (this case it will not be overriden)

By default, no context passes down. But you may specify:

```
{
  "type": "...",
  ...
  "defaults": {
    "child_default_field": "5",
    "another_child_default_field": {
      "some_key": "some_value",
      "some_other_key": "some_other_value"
    }
  },
  "context": {
    "parent_input_field": "child_field_to_set",
    "another_parent_input_field": "another_child_field_to_set",
  },
  "exportContext": {
    "field_to_set": "parent_model_field",
    "another_field_to_set": {
      "type": "...",
      ...
    },
  }
}
```

- "defaults" - set default values into descending json object
- "context" - copies field values of parent json object structure into child, the key is parent structure field, the value is child's (or they are same of you specify array without keys instead of object)
- "exportContext" - the most interesting one, it uses exporting logic to retrieve some values from the parent Eloquent Model instead of parent input. In this example, we specify the custom set of rules. You may also use an array without keys instead of an object to use some mappings from the section of parent model. Or you may specify empty value '' to use all the rules for the class of Model.

## Extra modifiers

You may specify an array of extra modifiers. They work with any mapping type, but it is up to mapping type to decide whether it will have effect. You may also use your own modifiers in custom mapping types. Several notable extra modifiers:

- "ahead" will massively load all the Models of the class transfered in the same transaction
- "dictionary" will massively load all the Models from database
- "read-only" will not include the Model in transaction and it will be not writeable on import
- "always-new" is the opposite of "read-only", the Model will never be read from database, and will be created on each import if there is input for it

## Extending MappingType class

You may also write your own extensions of mapping types. Just implementing abstract methods of MappingType class will give you something similar to "method" mapping, but using the class for the same purpose. But there are several useful methods to override that will give flexibility to use inspections from static analyser which is not possible with "method" mapping.

- **MappingType::getRelatedClass** override this to specify class your mapping type relates to. It should be Transactible Eloquent Model, though, and should have specified table.
- **MappingType::getAutoCountInspectionClass** you may override auto-count inspection to be able to calculate counts for the table on your own.
- **MappingType::importsRelation** if you have specified getRelatedClass, you may want to also state that it is imported, that way inspector and engine will look at its rules section after parsing the current mapping.
- **MappingType::isAutoCount** you can decide there, whether the auto-count is applicable to the mapping
- **MappingType::getKeyFromMapping** useful for dictionaries, it specifies key to use when searching for the model, this mapping type relates to.

## Writing custom static analyser inspections

It is possible to extend even further, when writing custom mapping type you may also provide it with some inspectiosn. For that you need to implement StaticRuleInspection interface. You can run them from DTODefs using:
```
(new StaticRuleAnalysis($this->rules, 'App\\Modles\\SomeRootClass', [array or one of StaticRuleInspection instances]))->performAnalysis();
```