<?php

namespace InSegment\ApiCore\Api\V_2_0;

use Closure;

use InSegment\ApiCore\Facades\DTODefs;
use InSegment\ApiCore\Interfaces\DTOCharacteristics;

use Illuminate\Support\Collection;
use Illuminate\Support\Str;

class DTOSettingsBuilder
{
    /**
     * RuleSetLoader instance
     *
     * @var \InSegment\ApiCore\Api\V_2_0\RuleSetLoader
     */
    protected $ruleSetLoader;
    
    /**
     * Options for DTO
     * 
     * @var array
     */
    protected $options = [
        'mode' => null,
        'dataCollection' => null,
        'resultAddress' => null,
        'resultKey' => null,
        'rootClass' => null,
        'rules' => [],
        'queryBuildingLayer' => null,
        'resultClass' => null,
        'validationRules' => null,
        'multiple' => null,
        'onProgress' => null,
        'onSuccess' => null,
        'chunkSize' => null,
        'transformations' => [],
        'keyBy' => null
    ];
    
    /**
     * Constructor
     * 
     * @param array $options
     */
    public function __construct(array $options = [])
    {
        $this->ruleSetLoader = DTODefs::getRuleSetLoader();
        
        foreach ($options as $key => $value) {
            if (array_key_exists($key, $this->options)) {
                $this->options[$key] = $value;
            }
        }
    }
    
    /**
     * Build DataTransferObject from the settings builder
     * 
     * @return \InSegment\ApiCore\Api\V_2_0\DataTransferObject
     */
    public function buildDTO(): DataTransferObject
    {
        return new DataTransferObject($this);
    }
    
    /**
     * Get options
     * 
     * @return array
     */
    public function getOptions(): array
    {
        return $this->options;
    }
    
    /**
     * Set root class of import
     * 
     * @param string $class
     * @return $this
     */
    public function setRootClass(string $class)
    {
        $this->options['rootClass'] = $class;
        return $this;
    }
    
    /**
     * Load rules from file
     * 
     * @param string $name
     * @return $this
     */
    public function loadRulesFile(string $name)
    {
        $this->options['rules'] = $this->ruleSetLoader->getRulesForIterator($name);
        return $this;
    }
    
    /**
     * Use custom rule set
     * 
     * @param \InSegment\ApiCore\Api\V_2_0\CustomRuleSet $ruleSet
     * @return $this
     */
    public function loadCustomRules(CustomRuleSet $ruleSet)
    {
        $this->ruleSetLoader->loadCustomRuleSet($ruleSet);
        $this->loadRulesFile($ruleSet->getName());
        return $this;
    }
    
    /**
     * Set the data collection for processing
     * 
     * @param \Illuminate\Support\Collection $collection
     * @return $this
     */
    public function setDataCollection(Collection $collection)
    {
        $this->options['dataCollection'] = $collection;
        return $this;
    }
    
    /**
     * Set the base address of result on data object
     * 
     * @param string $address
     * @return $this
     */
    public function setResultAddress($address)
    {
        $this->options['resultAddress'] = $address;
        return $this;
    }
    
    /**
     * Set the key of the model in the result object
     * 
     * @param array|string|null $key
     * @return $this
     */
    public function setResultKey($key)
    {
        $this->options['resultKey'] = $key;
        return $this;
    }
    
    /**
     * Set the key of the model in the result object
     * 
     * @param mixed $keyByRule
     * @return $this
     */
    public function keyBy($keyByRule)
    {
        $this->options['keyBy'] = $this->ruleSetLoader->processKeyByRule($keyByRule);
        return $this;
    }
    
    /**
     * Set query building layer
     * 
     * @param \Closure|callable $layer
     * @return $this
     */
    public function setQueryBuildingLayer($layer)
    {
        if (!$layer instanceof Closure) {
            $layer = Closure::fromCallable($layer);
        }
        
        $this->options['queryBuildingLayer'] = $layer;
        return $this;
    }
    
    /**
     * Set result class
     * 
     * @param string|\Illuminate\Database\Eloquent\Model $value
     */
    public function setResultClass($value)
    {
        if (is_string($value)) {
            $this->options['resultClass'] = $value;
        } else {
            $this->options['resultClass'] = get_class($value);
        }
        
        if (!isset($this->options['resultKey'])) {
            $this->setResultKey(DTODefs::getExternalKeyDefaultName($this->options['resultClass']));
        }
        
        return $this;
    }
    
    /**
     * Set whether there should be an array of objects in data (true), or just one object (false)
     * 
     * @param bool $multiple
     * @return $this
     */
    public function setMultiple(bool $multiple)
    {
        $this->options['multiple'] = $multiple;
        return $this;
    }
    
    /**
     * Add rules to validation
     * 
     * @param array $validationRules
     * @return $this
     */
    public function addRulesForValidation(array $validationRules)
    {
        $this->options['validationRules'] = array_merge($this->options['validationRules'], $validationRules);
        return $this;
    }
    
    /**
     * Set progress callback
     * 
     * @param Closure $onProgress
     * @return $this
     */
    public function setProgressCallback(Closure $onProgress)
    {
        $this->options['onProgress'] = $onProgress;
        return $this;
    }
    
    /**
     * Set success callback
     * 
     * @param Closure $onSuccess
     * @return $this
     */
    public function setSuccessCallback(Closure $onSuccess)
    {
        $this->options['onSuccess'] = $onSuccess;
        return $this;
    }
    
    /**
     * Set chunk size
     * 
     * @param int $size
     * @throws \Exception
     */
    public function setChunkSize(int $size = null)
    {
        if (isset($size) && $size <= 0) {
            throw new \Exception("Chunk size should be positive!");
        }
        
        $this->options['chunkSize'] = $size;
    }
    
    /**
     * Add result collection transformations
     * 
     * @param array $transformations
     */
    public function transformations(array $transformations)
    {
        $this->options['transformations'] = $transformations;
    }
    
    /**
     * Invoke entry point to configure DTOSettingsBulder
     * 
     * @param string $entryPointName
     * @throws \Exception
     * @return $this
     */
    public function invokeEntryPoint(string $entryPointName)
    {
        if (!isset($this->options['dataCollection'])) {
            throw new \Exception("DataCollection is required to invoke entry point with DTOSettingsBuilder!");
        }
        
        $dtoDefs = DTODefs::instance();
        if (($entryPoint = $dtoDefs->getEntryPointInstance($entryPointName, $this))) {
            switch ($this->options['mode']) {
                case DTOCharacteristics::MODE_UPDATE:
                    $this->setUpdate(true);
                    $entryPoint->update($this->options['dataCollection']);
                break;
                case DTOCharacteristics::MODE_IMPORT:
                    $entryPoint->import($this->options['dataCollection']);
                break;
                case DTOCharacteristics::MODE_EXPORT:
                    $entryPoint->export($this->options['dataCollection']);
                break;
                default: throw new \Exception("Unknown mode: {$this->options['mode']}");
            }
        } else {
            $this->setResultAddress(Str::singular(Str::snake($entryPointName)));
            
            switch ($this->options['mode']) {
                case DTOCharacteristics::MODE_UPDATE: $this->setUpdate(true); // fallthrough to import
                case DTOCharacteristics::MODE_IMPORT: $methodName = 'import' . ucfirst(Str::camel($entryPointName)); break;
                case DTOCharacteristics::MODE_EXPORT: $methodName = 'export' . ucfirst(Str::camel($entryPointName)); break;
                default: throw new \Exception("Unknown mode: {$this->options['mode']}");
            }
            
            if (method_exists($dtoDefs, $methodName)) {
                $dtoDefs->$methodName($this->options['dataCollection'], $this);
            } else {
                throw new \Exception("Entry point has no accociated class, and DTODefs have no method '{$methodName}'");
            }
        }
        
        return $this;
    }
}
