<?php

namespace InSegment\ApiCore\Api\V_2_0;

use Illuminate\Support\Str;
use InSegment\ApiCore\Api\V_1_0\DTOUtil as V1Util;
use InSegment\ApiCore\Api\V_2_0\RuleSetLoader;
use InSegment\ApiCore\Api\V_2_0\DTOSettingsBuilder;

use InSegment\ApiCore\Interfaces\EntryPoint;

use InSegment\ApiCore\Middleware\ChooseVersion;

abstract class DTOUtil extends V1Util
{
    /**
     * Application path to the version
     *
     * @var string
     */
    protected $appApiPath;
    
    /**
     * RuleSetLoader instance
     *
     * @var \InSegment\ApiCore\Api\V_2_0\RuleSetLoader 
     */
    protected $ruleSetLoader;
    
    /**
     * Entry point instances
     *
     * @var \InSegment\ApiCore\Interfaces\EntryPoint[] 
     */
    protected $entryPointInstances = [];
    
    /**
     * Entry points with class
     *
     * @var array [
     *     'entryPointName' => 'App\Api\V_1_0\EntryPoints\SomeEntryPoint',
     *     ...
     * ]
     */
    protected $entryPointsWithClass;
    
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->cachePrefix = config('api_core.cache_prefix', 'insegment.api-core.') . 'dto-utils-cache';
        
        $version = $this->getVersionString();
        $this->appApiPath = rtrim(ChooseVersion::getVersionsPath(), DIRECTORY_SEPARATOR);
        $uiidablesPath = implode(DIRECTORY_SEPARATOR, [$this->appApiPath, $version, 'uuidables.json']);
        $this->uuidables = $this->loadUUIDables($uiidablesPath);
        $this->ruleSetLoader = new RuleSetLoader($this);
        
        foreach ($this->getMappingTypeClasses() as $mappingTypeClass) {
            foreach ($mappingTypeClass::coversTypes() as $type) {
                $this->mappingTypeHandlers[$type] = $mappingTypeClass;
            }
        }
        
        $this->entryPointsWithClass = $this->getEntryPointsWithClass();
    }
    
    /**
     * Get RuleSetLoader instance
     * 
     * @return \InSegment\ApiCore\Api\V_2_0\RuleSetLoader
     */
    public function getRuleSetLoader()
    {
        return $this->ruleSetLoader;
    }
    
    /**
     * Load rules from file (will read it)
     * 
     * @param string $name
     * @return array|bool
     * @throws \Exception
     */
    public function loadRulesFile($name = 'rules')
    {
        if (!preg_match("/^[a-z0-9_-]+$/i", $name)) {
            throw new \Exception('Rules filename must match regular expression: "/^[a- z0-9_-]+$/i"');
        }
        
        $version = $this->getVersionString();
        $rulesPath = implode(DIRECTORY_SEPARATOR, [$this->appApiPath, $version, "{$name}.json"]);
        return $this->rules = $this->getRulesFileContents($rulesPath);
    }

    /**
     * Get entry points with separate classes
     * 
     * @return array [
     *     'entryPointName' => 'App\Api\V_1_0\EntryPoints\SomeEntryPoint',
     *     ...
     * ]
     */
    public function getEntryPointsWithClass(): array
    {
        return [];
    }
    
    /**
     * If exists, return instance of entry point with class
     * 
     * @param string $name
     * @param \InSegment\ApiCore\Api\V_2_0\DTOSettingsBuilder $settingsBuilder
     * @return \InSegment\ApiCore\Interfaces\EntryPoint|null
     */
    public function getEntryPointInstance(string $name, DTOSettingsBuilder $settingsBuilder)
    {
        if (isset($this->entryPointsWithClass[$name])) {
            if (!isset($this->entryPointInstances[$name])) {
                $class = $this->entryPointsWithClass[$name];
                $entryPoint = new $class();
                if (!$entryPoint instanceof EntryPoint) {
                    throw new \Exception("Entry point class must implement 'InSegment\ApiCore\Interfaces\EntryPoint' interface");
                }

                if (method_exists($entryPoint, 'getResultAddress')) {
                    $settingsBuilder->setResultAddress($entryPoint->getResultAddress());
                } else {
                    $settingsBuilder->setResultAddress(Str::singular(Str::snake($name)));
                }

                $entryPoint->configure($settingsBuilder);
                $this->entryPointInstances[$name] = $entryPoint;
            }
            
            return $this->entryPointInstances[$name];
        }
    }
    
    /**
     * Get known mapping type classes
     * 
     * @return array
     */
    public function getMappingTypeClasses()
    {
        return [
            MappingTypes\Attribute::class,
            MappingTypes\Relation::class,
            MappingTypes\Method::class
        ];
    }
    
    public function findObjectsForExport(\InSegment\ApiCore\Api\DataTransferObject $dto): \Illuminate\Database\Eloquent\Collection
    {
        throw new \Exception("The method is depreacted in V_2_0");
    }

    public function findObjectsForImport(\InSegment\ApiCore\Api\DataTransferObject $dto)
    {
        throw new \Exception("The method is depreacted in V_2_0");
    }

}
