<?php

namespace InSegment\ApiCore\Api\V_2_0;

use Illuminate\Support\Facades\Validator;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;

use InSegment\ApiCore\Services\Transactor;
use InSegment\ApiCore\Facades\DTODefs;

class DataTransferObject
{
    const DEFAULT_CHUNK_SIZE = 100;
    
    /**
     * Mode of transfer operation
     *
     * @var int
     */
    protected $mode;
    
    /**
     * Address of result in the data collection
     *
     * @var string
     */
    protected $resultAddress;
    
    /**
     * Key of result in the data collection
     *
     * @var array|string|null
     */
    protected $resultKey;
    
    /**
     * Root class
     *
     * @var string
     */
    protected $rootClass;
    
    /**
     * Result class
     *
     * @var string 
     */
    protected $resultClass;
    
    /**
     * Rules for classes
     * 
     * @var array
     */
    protected $rules;
    
    /**
     * Building layer for queries for models
     *
     * @var \Closure
     */
    protected $queryBuildingLayer;
    
    /**
     * Rules to validate on data
     *
     * @var array
     */
    protected $validationRules = [];
    
    /**
     * Whether there should be an array of data (true) or just one object (false)
     *
     * @var bool 
     */
    protected $multiple = false;
    
    /**
     * Transformations for results collection
     *
     * @var array
     */
    protected $transformations = [];
    
    /**
     * Active chunk size
     *
     * @var int 
     */
    protected $chunkSize = self::DEFAULT_CHUNK_SIZE;
    
    /**
     * On progress callback
     * 
     * @var \Closure
     */
    protected $onProgress;
    
    /**
     * On success callback
     * 
     * @var \Closure
     */
    protected $onSuccess;
    
    /**
     * Data collection
     *
     * @var \Illuminate\Support\Collection
     */
    protected $dataCollection;
    
    /**
     * DTODefs instance
     * 
     * @var \InSegment\ApiCore\Api\V_2_0\DTOUtil
     */
    protected $dtoDefs;
    
    /**
     * Validator
     *
     * @var \Illuminate\Contracts\Validation\Validator
     */
    protected $validator;
    
    /**
     * Dictionary mapping data keys to external keys
     *
     * @var \InSegment\ApiCore\Api\V_2_0\DTODictionary
     */
    protected $dictionary;
    
    /**
     * Query processor instance
     *
     * @var \InSegment\ApiCore\Api\V_2_0\TransferQueryProcessor|null
     */
    protected $queryProcessor;
    
    /**
     * Key for data result on export
     *
     * @var array|null
     */
    protected $keyBy;
    
    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Api\V_2_0\DTOSettingsBuilder $optionsBuilder
     */
    public function __construct(DTOSettingsBuilder $optionsBuilder)
    {
        $this->dtoDefs = DTODefs::instance();
        $haveOptions = $optionsBuilder->getOptions();
        $missingOptions = [];
        $needOptions = [
            'mode' => true,
            'dataCollection' => true,
            'transformations' => true,
            'resultAddress' => true,
            'resultKey' => false,
            'rootClass' => true,
            'rules' => true,
            'queryBuildingLayer' => true,
            'resultClass' => true,
            'validationRules' => false,
            'multiple' => true,
            'onProgress' => false,
            'onSuccess' => false,
            'chunkSize' => false,
            'keyBy' => false
        ];

        foreach ($needOptions as $key => $isNecessary) {
            if (!isset($haveOptions[$key])) {
                if ($isNecessary) {
                    $missingOptions[] = $key;
                }
            } else {
                $this->$key = $haveOptions[$key];
            }
        }
        
        if ($missingOptions) {
            throw new \Exception("DataTransferObject: missing required options " . implode(', ', $missingOptions));
        }
    }
    
    /**
     * Get mode of transfer operation
     * 
     * @return int
     */
    public function getMode()
    {
        return $this->mode;
    }
    
    /**
     * Get root class
     * 
     * @return string|null
     */
    public function getRootClass()
    {
        if (!isset($this->rootClass)) {
            throw new \Exception('The DTO root class is not set');
        }
        
        return $this->rootClass;
    }
    
    /**
     * Get rules
     * 
     * @return array [
     *     'classToAttribute' => array,
     *     'mapperTypes' => array [
     *          string $type => string $class,
     *          ...
     *     ]
     * ]
     */
    public function getRules(): array
    {
        return $this->rules;
    }
    
    /**
     * Get result address
     * 
     * @return string
     */
    public function getResultAddress()
    {
        return $this->resultAddress;
    }
    
    /**
     * Get the key of the model in the result object
     * 
     * @return array|string
     */
    public function getResultKey()
    {
        return $this->resultKey;
    }
    
    /**
     * Get key for data on export
     * 
     * @return array [
     *     'type' => 'keys'|'closure'|'method',
     *     'value' => array|\Closure|callable
     * ]|null
     */
    public function getKeyBy()
    {
        return $this->keyBy;
    }
    
    /**
     * Get the class of the model for the result object
     * 
     * @return string
     */
    public function getResultClass()
    {
        return $this->resultClass;
    }
    
    /**
     * Get query building layer
     * 
     * @return \Closure
     */
    public function getQueryBuildingLayer(): Closure
    {
        return $this->queryBuildingLayer;
    }
    
    /**
     * Get query from query building layer
     * 
     * @param array $buildingLayerParams
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getQuery(...$buildingLayerParams): Builder
    {
        return ($this->queryBuildingLayer)(...$buildingLayerParams);
    }

    /**
     * Whether there should be an array of objects in data (true), or just one object (false)
     * 
     * @return bool
     */
    public function getIsMultiple()
    {
        return $this->multiple;
    }
    
    /**
     * Get collection
     * 
     * @return \Illuminate\Support\Collection
     */
    public function getCollection()
    {
        return $this->dataCollection;
    }
    
    /**
     * Get validator for data
     * 
     * @return \Illuminate\Contracts\Validation\Validator
     */
    public function getValidator()
    {
        if (!isset($this->validator)) {
            $this->validator = Validator::make($this->dataCollection->all(), $this->validationRules);
        }
        
        return $this->validator;
    }
    
    /**
     * Get dictionary for results
     * 
     * @return \InSegment\ApiCore\Api\V_2_0\DTODictionary
     */
    public function getDictionary()
    {
        if (!isset($this->dictionary)) {
            $this->dictionary = new DTODictionary(
                $this->dtoDefs,
                $this->getResultClass(),
                $this->getResultKey(),
                $this->getData(),
                ['multiple' => $this->getIsMultiple()]
            );
        }
        
        return $this->dictionary;
    }
    
    /**
     * Get data from result by key
     * 
     * @param string|null $key
     * @return mixed
     */
    public function getData(string $key = null)
    {
        if (isset($key)) {
            $address = "{$this->getResultAddress()}.{$key}";
        } else {
            $address = $this->getResultAddress();
        }
        
        return $this->dataCollection->get($address);
    }
    
    /**
     * Get transformations for result collection
     * 
     * @return array
     */
    public function getTransformations()
    {
        return $this->transformations;
    }
    
    /**
     * Set the object to the Transactor session
     * 
     * @param mixed $object
     * @return $this
     */
    public function setSession(string $class, $value)
    {
        Transactor::setSessionObject($class,
            ($value instanceof Model
                ? $value->getKey()
                : ($value instanceof EloquentCollection
                    ? $value->pluck((new $class)->getKeyName())
                    : $value
                ))
        );
        
        return $this;
    }
    
    /**
     * Give the required objects stored in the Transactor session if they are present, or throw an Exception, if not
     * 
     * @param array|string $sessionRequirements
     * @return mixed
     */
    public function requireSession($sessionRequirements)
    {
        $isArray = is_array($sessionRequirements);
        $result = $isArray ? [] : null;
        
        foreach (($isArray ? $sessionRequirements : [$sessionRequirements]) as $class) {
            $object = Transactor::getSessionObject($class);
            if (!isset($object)) {
                throw new \Exception("The required object of class '{$class}' is not set in the session");
            }
            
            if ($isArray) {
                $result[$class] = $object;
            } else {
                $result = $object;
            }
        }
        
        return $result;
    }
    
    /**
     * Get chunk size for query
     * 
     * @return int
     */
    public function getChunkSize()
    {
        return $this->chunkSize;
    }
    
    /**
     * Process DataTransferObject
     * 
     * @param array $processorOptions
     * @param array $params
     * @return \InSegment\ApiCore\Api\V_2_0\TransferQueryProcessor
     */
    public function process(array $processorOptions = [], ...$buildingLayerParams)
    {
        if (!isset($this->queryProcessor)) {
            $this->queryProcessor = new TransferQueryProcessor($this);
        }
        
        return $this->queryProcessor->process($this->getQuery(...$buildingLayerParams), $processorOptions);
    }
    
}
