<?php

namespace InSegment\ApiCore\Api\V_2_0\MappingTypes;

use Illuminate\Support\Str;
use InSegment\ApiCore\Models\TraverseState;
use InSegment\ApiCore\Interfaces\DTOCharacteristics;

class Attribute extends MappingType
{
    const MAPPED_DATA = 1 << 0;
    const MAPPED_CAST = 1 << 1;
    const MAPPED_FUN = 1 << 2;
    
    const CAST_ARRAY = 0;
    const CAST_STRING = 1;
    const CAST_DOUBLE = 2;
    const CAST_FLOAT = 3;
    const CAST_INT = 4;
    
    /**
     * Should return an array of strings representing types of mappings
     * which are to be handled by this class
     * 
     * @return array
     */
    public static function coversTypes(): array
    {
        return ['attribute'];
    }

    /**
     * Import
     * 
     * @param \InSegment\ApiCore\Models\TraverseState $state
     */
    public function import(TraverseState $state)
    {
        $attribute = $state->mapping['attribute'];

        if (isset($state->mapping['mapped'])) {
            $target = $state->model;
            $setTarget = $target;
            $setAddress = [];
            $data = $state->data;
            
            foreach ($state->mapping['mapped'] as $step) {
                if (!isset($target)) {
                    break;
                }
                
                if (isset($data) && $step['characteristic'] & self::MAPPED_CAST) {
                    switch($step['castType']) {
                        case self::CAST_ARRAY: $data = (array) $data; break;
                        case self::CAST_STRING: $data = (string) $data; break;
                        case self::CAST_DOUBLE: $data = (double) $data; break;
                        case self::CAST_FLOAT: $data = (float) $data; break;
                        case self::CAST_INT: $data = (int) $data; break;
                        default: throw new \Exception('Uknown cast type: ' . $step['castType']);
                    }
                }
                
                if ($step['characteristic'] & self::MAPPED_FUN) {
                    $funTarget = data_get($target, implode('.', $setAddress));
                    $setTarget = $target = $funTarget->{$step['part']}($data);
                    $setAddress = [];
                } else if ($step['characteristic'] & self::MAPPED_DATA) {
                    $setAddress[] = $step['part'];
                }                
            }

            if ($setAddress) {
                data_get($setTarget, implode('.', $setAddress), $data, $state->mode === DTOCharacteristics::MODE_UPDATE);
            }
        } else if ($state->mode === DTOCharacteristics::MODE_UPDATE || !isset($state->model->$attribute)) {
            $state->model->setAttribute($attribute, $state->data);
        }
    }

    /**
     * Export
     * 
     * @param \InSegment\ApiCore\Models\TraverseState $state
     */
    public function export(TraverseState $state)
    {
        if (isset($state->mapping['mapped'])) {
            $target = $state->model;
            
            foreach ($state->mapping['mapped'] as $step) {
                if (!isset($target)) {
                    break;
                }
                
                if ($step['characteristic'] & self::MAPPED_FUN) {
                    $target = is_string($target) ? $target : $target->{$step['part']}();
                } else if ($step['characteristic'] & self::MAPPED_DATA) {
                    $target = data_get($target, $step['part']);
                }
                
                if ($step['characteristic'] & self::MAPPED_CAST) {
                    switch($step['castType']) {
                        case self::CAST_ARRAY: $target = (array) $target; break;
                        case self::CAST_STRING: $target = (string) $target; break;
                        case self::CAST_DOUBLE: $target = (double) $target; break;
                        case self::CAST_FLOAT: $target = (float) $target; break;
                        case self::CAST_INT: $target = (int) $target; break;
                        default: throw new \Exception('Uknown cast type: ' . $step['castType']);
                    }
                }
            }

            $state->data = $target;
        } else {
            $state->data = $state->model->getAttribute($state->mapping['attribute']);
        }
    }
    
    /**
     * Configure attribute rule mapping
     * 
     * @param string $class
     * @param array $mapping
     * @return array
     */
    public static function configureRuleMapping($class, $mapping): array
    {
        if (!is_array($mapping)) {
            $mapping = ['type' => 'attribute', 'attribute' => $mapping];
        }
        
        if (Str::contains($mapping['attribute'], ['.', ':', '()'])) {
            $mapping['mapped'] = static::characterizeAttributeMapping($mapping['attribute']);
        }
        
        return $mapping;
    }

    /**
     * Export field mapping, supporting calling and casting, and dot notation
     * 
     * @param string $mapping
     * @return null
     */
    public static function characterizeAttributeMapping(string $mapping)
    {
        $parts = [];
        
        foreach (explode('.', $mapping) as $part) {
            /**
             * $part Variants:
             * 'somefield',
             * 'somefield:somecast',
             * 'somemethod()'
             * 'somemethod():somecast'
             */
            
            if (Str::endsWith($part, '()')) {
                $part = substr($part, 0, -2);
                $parts[] = ['part' => $part, 'characteristic' => self::MAPPED_FUN]; // just 'somemethod()'
            } else {
                $explode = explode(':', $part, 2);
                $part = $explode[0];
                $cast = $explode[1] ?? null;
            
                if (isset($cast)) {
                    switch($cast) {
                        case 'array': $castType = self::CAST_ARRAY; break;
                        case 'string': $castType = self::CAST_STRING; break;
                        case 'double': $castType = self::CAST_DOUBLE; break;
                        case 'float': $castType = self::CAST_FLOAT; break;
                        case 'int': $castType = self::CAST_INT; break;
                        default: throw new \Exception('Unknown cast in attribute mapping: ' . $mapping);
                    }
            
                    if (Str::endsWith($part, '()')) {
                        $part = substr($part, 0, -2);
                        $parts[] = ['part' => $part, 'characteristic' => self::MAPPED_CAST|self::MAPPED_FUN, 'castType' => $castType]; // 'somemethod():somecast'
                    } else {
                        $parts[] = ['part' => $part, 'characteristic' => self::MAPPED_CAST|self::MAPPED_DATA, 'castType' => $castType]; // 'somefield:somecast'
                    }
                } else {
                    $parts[] = ['part' => $part, 'characteristic' => self::MAPPED_DATA]; // just 'somefield'
                }
            }
        }
        
        return $parts;
    }
    
}
