<?php

namespace InSegment\ApiCore\Api\V_2_0\MappingTypes;

use Illuminate\Support\Arr;
use InSegment\ApiCore\Models\TraverseState;
use InSegment\ApiCore\Services\Transactor;
use InSegment\ApiCore\Interfaces\DTOCharacteristics;
use Illuminate\Database\Eloquent\Collection;
use InSegment\ApiCore\Models\SADSource;

use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Relation extends MappingType
{
    const SEARCH_MODE_NORMAL = 0;
    const SEARCH_MODE_NEW = 1;
    const SEARCH_MODE_FRESH = 2;
    
    /**
     * Dictionaries and models preloaded ahead in transaction loaded for classes
     * 
     * @var array
     */
    protected $preloads = [];
    
    /**
     * Should return an array of strings representing types of mappings
     * which are to be handled by this class
     * 
     * @return array
     */
    public static function coversTypes(): array
    {
        return ['one', 'many', 'find'];
    }

    /**
     * Configure rule mapping
     * 
     * @param string $class
     * @param array $mapping
     * @return array
     */
    public static function configureRuleMapping($class, $mapping): array
    {
        $example = new $class();
        $relationName = $mapping['relation'];
        $relation = $example->$relationName();
        $mapping['isDownstream'] = !$relation instanceof BelongsTo;
        if ($mapping['isDownstream']) {
            if ($mapping['type'] === 'find') {
                throw new \Exception(
                    "Problem in relation mapping for {$relationName} of {$class}:"
                    . " Find type of relation can only be used with upstream relations (BelongsTo)"
                );
            }
        }
            
        $relation->initRelation([$example], $relationName);
        $relationValue = $example->getRelationValue($relationName);
        $mapping['isCollection'] = $relationValue instanceof Collection;
        
        return $mapping;
    }
    
    /**
     * Import
     * 
     * @param \InSegment\ApiCore\Models\TraverseState $state
     */
    public function import(TraverseState $state)
    {
        if (!isset($state->data)) {
            return;
        }
        
        if ($state->mapping['isDownstream']) {
            $this->nonRecursiveIterator->addDownStream(['closure' => function (TraverseState $state) {
                $this->relateDownstreamRelation($state);
            }, 'moment' => $state->getMoment()]);
        } else {
            $this->relateUpstreamRelation($state);
        }
    }

    /**
     * Export
     * 
     * @param \InSegment\ApiCore\Models\TraverseState $state
     */
    public function export(TraverseState $state)
    {
        $relationName = $state->mapping['relation'];
        $related = $state->model->getRelationValue($relationName);
        $relatedClass = static::getRelatedClass($state->mapping);
        $multiple = false;
        
        if ($related === null) {
            return;
        }
        
        if ($state->mapping['isDownstream']) {
            if ($state->mapping['isCollection']) {
                $source = $related->values()->all();
                $multiple = true;
            } else {
                $source = [$related];
            }
        } else if (!isset($state->mapping['extra']['show']) && is_string($extKey = static::getKeyFromMapping($state->mapping))) {
            $state->data = $related->getAttribute($extKey);
            return;
        } else {
            $source = [$related];
        }
        
        $attribute = $this->queryProcessor->classToAttributes[$relatedClass];
        $this->nonRecursiveIterator->newSource(new SADSource($attribute, $source, [
            'keyBy' => $state->mapping['keyBy'] ?? null,
            'multiple' => $multiple,
            'dataAddress' => $state->mapping['dataAddress'],
            'class' => $relatedClass,
            'relation' => null,
            'relationName' => $relationName
        ]));
    }
    
    protected function relateDownstreamRelation(TraverseState $state)
    {
        $searchMode =
            (isset($state->mapping['extra']['fresh']) ? self::SEARCH_MODE_FRESH : null)
            ?? (isset($state->mapping['extra']['always-new']) ? self::SEARCH_MODE_NEW : null)
            ?? self::SEARCH_MODE_NORMAL;
                        
        switch ($state->mapping['type']) {
            case 'one':
                $relationName = $state->mapping['relation'];
                $relation = $state->model->$relationName();
                $related = $this->dtoDefs->getOneRelated($state->model, $relationName, $searchMode);
                $state->model->setRelation($relationName, $related);
                // $this->importModel(['relation' => $relation, 'model' => $model, 'dotFormat' => $dotFormat, 'mapping' => $mapping, 'context' => $context, 'omit' => true]);
            break;
            case 'many':
                $this->importDescendants(['relation' => $relation, 'dotFormat' => $dotFormat, 'mapping' => $mapping], $relationName, $searchMode);
            break;
            case 'find': throw new \Exception("Find type of relation can only be used with upstream relations (BelongsTo)");
        }
    }

    /**
     * Import Models which are dependent on the currently previous-in-stack Model
     * 
     * @param \InSegment\ApiCore\Models\TraverseState $state
     * @param int $searchMode
     * @throws \Exception
     */
    protected function importDescendants(TraverseState $state, int $searchMode)
    {
        $relationName = $state->mapping['relation'];
        $relatedClass = static::getRelatedClass($state->mapping);
        $extKey = static::getKeyFromMapping($state->mapping) ?? $this->dtoDefs->getExternalKeyDefaultName($relatedClass);
        
        // if the external key is defined, we can get items in any order, with any gaps
        if ($extKey) {
            $relInfo = $this->dtoDefs->mapKeysByInput($relatedClass, $state->data) + ['relation' => $relationName, 'searchMode' => $searchMode, 'externalKey' => $extKey];
            
            // get existing and new objects, new objects are last ones
            $relatedPlusNew = $this->dtoDefs->getManyRelatedWithKeys($state->model, $relInfo, count($state->data));
            $state->model->setRelation($relationName, $relatedPlusNew);
            
            // each found object certainly have data for it, since it is joined with buffer
            // $this->importManyWithKeys($relatedPlusNew, $extKey, $relInfo['inputKeyByExtKeys'], true);
        } else {
            // otherwise work in the correct order
            // get maximum of so much records, that we've received
            
            $relatedPlusNew = $this->dtoDefs->getManyRelated($state->model, $relationName, count($state->data), $searchMode);
            $state->model->setRelation($relationName, $relatedPlusNew);
            
            // clone relation to avoid setting wrong restrictions on original relation object
            /* @see EloquentServiceProvider */
            $this->import($relatedPlusNew);
        }
    }
    
    /**
     * Relate BelongsTo relation. Handles cases such is 'only id given',
     * found or not found, import if given the whole object, et cetera...
     * 
     * @param \InSegment\ApiCore\Models\TraverseState $state
     * @throws \Exception
     */
    protected function relateUpstreamRelation(TraverseState $state)
    {
        $relationName = $state->mapping['relation'];
        $relation = $state->model->$relationName();
        $relatedClass = static::getRelatedClass($state->mapping);
        $extKeysInfo = static::getExtKeysInfo($this->dtoDefs, $state->mapping, $state->data);
        $alwaysNew = isset($state->mapping['extra']['always-new']);

        // Use eager load if possible, ignore eager load when the model should always be new
        if (!$alwaysNew && $state->model->relationLoaded($relationName)) {
            $related = $state->model->getRelation($relationName);
        }

        // state in that variable, whether we don't have the model from eager load
        // eager loaded models should avoid checks on DTOCharacteristics::DATA_GIVEN_NOT_ENOUGH,
        // because they are not affected by input
        $findRelated = !isset($related);

        if ($findRelated) {
            $related = $this->findOrMakeChild($state, $relation, $relatedClass, $extKeysInfo, $alwaysNew);
        }

        $dataCharacteristic = $this->dtoDefs->characterizeGivenData($relatedClass, $state->data, $related->exists, $extKeysInfo);

        // $findRelated means that model didn't come from eager load
        // if there are ext keys specified in rules mapped to values present or absent in input
        // if characterized as not enough data and do not exists already
        if ($findRelated && count($extKeysInfo) && $dataCharacteristic === DTOCharacteristics::DATA_GIVEN_NOT_ENOUGH && !$related->exists) {
            $debug = json_encode(['keysInfo' => $extKeysInfo, 'class' => $relatedClass, 'data' => $state->data], JSON_PARTIAL_OUTPUT_ON_ERROR);
            throw new \Exception("No data is given for a key of '{$relatedClass}' which is not present in the database or in the transaction. Debug: {$debug}");
        }

        // Import the object like any other if the create/update data is present, that means conditions:
        // 1. mapping imports relations ('find' for example does not)
        // 2. $related does not exists OR the input has more data than DATA_GIVEN_NOT_ENOUGH or DATA_GIVEN_KEY_NO_CONTENT
        if (
            static::importsRelation($state->mapping)
            && (
                !in_array($dataCharacteristic, [DataTransferObject::DATA_GIVEN_KEY_NO_CONTENT, DataTransferObject::DATA_GIVEN_NOT_ENOUGH])
                || !$related->exists
            )
        ) {
            // TODO: Add specific handling for MorphTo Relations
            // When a Model is always new, there may be contradiction on relation on its primary key, avoid it
            $query = $relation->getQuery()->getQuery();
            $keyName = $relation->getQualifiedOwnerKeyName();
            foreach($query->wheres as &$value) {
                if ($value['column'] === $keyName) {
                    if ($alwaysNew && $value['type'] === 'Basic' && $value['operator'] === '=') {
                        $value['operator'] = '!=';
                    } else if ($findRelated && $value['type'] === 'Null') {
                        $value['type'] = 'NotNull';
                    }
                    break;
                }
            }

            $attribute = $this->queryProcessor->classToAttributes[$relatedClass];
            $this->nonRecursiveIterator->newSource(new SADSource($attribute, [$related], [
                'keyBy' => $state->mapping['keyBy'] ?? null,
                'multiple' => false,
                'dataAddress' => $state->mapping['dataAddress'],
                'class' => $relatedClass,
                'parent' => $state->model,
                'relation' => $relation,
                'relationName' => $relationName,
                'associate' => true
            ]));
        } else {
            // Actually, sets $state->model->field_in_question to the key of the $related model
            $relation->associate($related);
            $state->model->setRelation($relationName, $related);
        }
    }
    
    /**
     * This method works with BelongsTo relations. It is different from Has* relations by the means of
     * creating the related model before the parent model is saved, and associating it to the unsaved parent.
     * The default expected usage is:
     * given id, find the model or create if not found and associate it with parent;
     * given an object, find the model or create it by id from object, set necessary fields, and associate it with parent;
     * 
     * @param \InSegment\ApiCore\Models\TraverseState $state
     * @param \Illuminate\Database\Eloquent\Relations\BelongsTo $relation
     * @param string $relatedClass
     * @param array $extKeysInfo
     * @param bool $alwaysNew
     * @throws \Exception
     */
    protected function findOrMakeChild(TraverseState $state, BelongsTo $relation, $relatedClass, $extKeysInfo, $alwaysNew)
    {
        $command = $alwaysNew ? 'newModelInstance' : 'firstOrNew';
        
        $query = clone $relation;
        $baseQuery = $query->getBaseQuery();
        $baseQuery->wheres = [];
        $baseQuery->bindings['where'] = [];
        
        if (count($extKeysInfo)) {
            $suppliesForUniques = [];
//            if (isset($mapping['uniqueSupply'])) {
//                $this->exportModel(['input' => $suppliesForUniques, 'dotFormat' => '-', 'omit' => true], $mapping['uniqueSupply']);
//            }
            
            $searchAttributes = $this->dtoDefs->uniquesToAttributes($suppliesForUniques, $relatedClass, $extKeysInfo);
            
            $related = $this->getFromPreload($relatedClass, $extKeysInfo, $searchAttributes);
            if (!isset($related)) {
                $related = $query->$command($searchAttributes);
            }
            
            return $related;
        } else {
            $checkable = $this->dtoDefs->collectCheckableAttributes($relatedClass, $state->data);
            return $query->$command($checkable);
        }
    }
    
    /**
     * Get Model from preload, if it is there
     * 
     * @param string $class
     * @param array $extKeysInfo
     * @param array $searchAttributes
     * @return \Illuminate\Database\Eloquent\Model|null
     */
    private function getFromPreload($class, $extKeysInfo, $searchAttributes)
    {
        !isset($this->preloads[$class]) && ($this->preloads[$class] = []);
        $extKeysImplode = implode(',', array_keys($extKeysInfo));
        
        if (!array_key_exists($extKeysImplode, $this->preloads[$class])) {
            $preloadClosure = Transactor::getPreloads($class, $extKeysImplode);
            $this->preloads[$class][$extKeysImplode] = isset($preloadClosure) ? $preloadClosure() : null;
        }
        
        if (isset($this->preloads[$class][$extKeysImplode])) {
            if (($group = $this->preloads[$class][$extKeysImplode]->get(implode(',', array_values($extKeysInfo))))) {
                foreach ($searchAttributes as $key => $value) {
                    $group = $group->where($key, $value);
                }
                
                return $group->first();
            }
        }
        
        return null;
    }
    /**
     * ==========================================
     * STUFF FOR STATIC ANALYSIS | TODO: REFACTOR
     * ==========================================
     */
    
    /**
     * Enumerate relation type, return string with full new address if enumeration is to recurse
     * 
     * @param array $output
     * @param string $address
     * @param string $class
     * @param string $dotFormat
     * @param mixed $mapping
     * @return bool
     */
    public static function enumerateRelationType(&$output, $address, $class, $dotFormat, &$mapping): bool
    {
        $fullAddress = $dotFormat === '-' ? $address : "{$address}.{$dotFormat}";
        $relationTypeName = basename(get_class(with(new $class)->{$mapping['relation']}()));

        $output[$fullAddress] = isset($output[$fullAddress])
            ? "{$output[$fullAddress]} => {$relationTypeName} {$mapping['class']}"
            : "{$relationTypeName} {$mapping['class']}";

        if (isset($mapping['extra'])) {
            $output[$fullAddress] .= ' (' . implode(', ', $mapping['extra']) . ')';
        }
        
        return static::importsRelation($mapping);
    }
    
    /**
     * Get related class for relation of mapping type
     * 
     * @param array $mapping
     * @return string
     */
    public static function getRelatedClass($mapping)
    {
        return $mapping['class'];
    }
    
    /**
     * Whether this MappingType relation should look for an input to process its object
     * 
     * @param array $mapping
     * @return bool
     */
    public static function importsRelation($mapping): bool
    {
        return $mapping['type'] !== 'find';
    }
    
    /**
     * Check whether the mapping type is auto-countable (one-to-one with parent)
     * 
     * @param array $mapping
     * @return bool
     */
    public static function isAutoCount($mapping): bool
    {
        return $mapping['type'] === 'one';
    }
    
    /**
     * Get key from mapping
     * 
     * @param array $mapping
     * @return array|string|null
     */
    protected static function getKeyFromMapping($mapping)
    {
        // for the 'find' type, prefer the specified key for search (maybe null) to default external key name
        return $mapping['key'] ?? null;
    }
    
    /**
     * Check whether records from the table of the related model could be deleted
     * 
     * @param array $mapping
     * @return bool
     */
    public static function isDeletionEnabled($mapping): bool
    {
        return in_array('fresh', Arr::get($mapping, 'extra', []));
    }

}
