<?php

namespace InSegment\ApiCore\Api\V_2_0;

use Illuminate\Support\Arr;
use InSegment\ApiCore\Api\V_2_0\DTOUtil;
use Illuminate\Support\Facades\Cache;

class RuleSetLoader
{
    /**
     * DTODefs instance
     *
     * @var \InSegment\ApiCore\Api\V_2_0\DTOUtil 
     */
    protected $dtoDefs;
    
    /**
     * Raw loaded rule sets
     *
     * @var array
     */
    protected $loadedRuleSets = [];
    
    /**
     * Cache of initialised rule sets
     *
     * @var array
     */
    protected $parsedRuleSets = [];
    
    /**
     * Cache prefix
     *
     * @var string
     */
    protected $cachePrefix;
    
    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Api\V_2_0\DTOUtil $dtoDefs
     */
    public function __construct(DTOUtil $dtoDefs)
    {
        $this->dtoDefs = $dtoDefs;
        $this->cachePrefix = config('api_core.cache_prefix', 'insegment.api-core.') . 'rule-sets-cache';
    }
    
    /**
     * Load rules from file using name
     * 
     * @param string $name
     * @return $this
     * @throws \Exception
     */
    public function loadRules(string $name)
    {
        if (!isset($this->loadedRuleSets[$name])) {
            $this->loadedRuleSets[$name] = $this->dtoDefs->loadRulesFile($name);
            
            if (!is_array($this->loadedRuleSets[$name])) {
                throw new \Exception("Failed to load rule set: {$name}");
            }
        }
        
        return $this;
    }
    
    /**
     * Load custom rules set into virtual rule set name
     * 
     * @param \InSegment\ApiCore\Api\V_2_0\CustomRuleSet $customRuleSet
     * @return null
     */
    public function loadCustomRuleSet(CustomRuleSet $customRuleSet)
    {
        $name = $customRuleSet->getName();
        $this->loadedRuleSets[$name] = $customRuleSet->rules;
    }
    
    /**
     * Get raw rule set (non-initialised for iterator)
     * 
     * @param string $name
     * @return array
     */
    public function getRuleSet(string $name)
    {
        $this->loadRules($name);
        
        return $this->loadedRuleSets[$name];
    }
    
    /**
     * Get rules for iterator (initialised)
     * 
     * @param string $name
     * @return array
     */
    public function getRulesForIterator($name)
    {
        if (!isset($this->parsedRuleSets[$name])) {
            $this->parsedRuleSets[$name] = $this->initialiseRulesForIterator($name);
        }
        
        return $this->parsedRuleSets[$name];
    }
    
    /**
     * Determine the type of keyBy rule and extract a Clousre, if possible
     * 
     * @param mixed $keyBy
     * @return array [
     *     'type' => 'keys'|'closure'|'method',
     *     'value' => array|\Closure|callable
     * ]
     */
    public function processKeyByRule($keyBy)
    {
        if (is_array($keyBy)) {
            return ['type' => 'keys', 'value' => $keyBy];
        } else if (is_callable($keyBy)) {
            return ['type' => 'callable', 'value' => implode('@', $keyBy)];
        } else if (strpos($keyBy, '@') !== false) {
            list($class, $method) = explode('@', $keyBy);
            
            if ($class) {
                return ['type' => 'callable', 'value' => $keyBy];
            } else {
                return ['type' => 'method', 'value' => $method];
            }
        } else if (is_string($keyBy)) {
            return [
                'type' => 'keys',
                'value' => [$keyBy]
            ];
        }
    }
    
    /**
     * Prepare rule set for file name
     * 
     * @param string $name
     * @return array [
     *     'classToAttribute' => array [
     *         string $class => [
     *              string $dotFormat => array [
     *                  'type' => 'mapperType',
     *                  ...
     *              ],
     *              ...
     *         ], ...
     *     ],
     *     'mapperTypes' => [
     *         string $mapperType => string $handlerClass,
     *         ...
     *     ]
     * ]
     * @throws \Exception
     */
    protected function initialiseRulesForIterator(string $name): array
    {
        $ruleSet = $this->getRuleSet($name);
        return Cache::rememberForever("{$this->cachePrefix}.{$name}", function () use ($ruleSet, $name) {
            $result = [];
            $mapperTypes = [];

            foreach ($ruleSet as $class => $mappings) {
                $notArray = !is_array($mappings);
                $attributeSet = [];
                $downstreamSet = [];
                foreach (Arr::wrap($mappings) as $dataAddress => $mapping) {
                    if (!isset($mapping)) {
                        continue;
                    }

                    if (is_array($mapping)) {
                        if (!isset($mapping['type'])) {
                            throw new \Exception("Malformed rule '{$dataAddress}' of '{$class}': 'type' field is required");
                        }

                        $mappingType = $mapping['type'];
                    } else {
                        $mappingType = 'attribute';
                    }

                    if (!isset($mapperTypes[$mappingType])) {
                        $mapperTypes[$mappingType] = $this->dtoDefs->classifyMapperType($mappingType);
                    }

                    $modifiedMapping = $mapperTypes[$mappingType]::configureRuleMapping($class, $mapping);
                    $modifiedMapping['extra'] = array_flip($modifiedMapping['extra'] ?? []);
                    $modifiedMapping['dataAddress'] = $notArray ? '-' : explode('.', $dataAddress);

                    if (isset($modifiedMapping['keyBy'])) {
                        $modifiedMapping['keyBy'] = $this->processKeyByRule($modifiedMapping['keyBy']);
                    }

                    if ($modifiedMapping['isDownstream'] ?? false) {
                        $downstreamSet[] = $modifiedMapping;
                    } else {
                        $attributeSet[] = $modifiedMapping;
                    }
                }

                $result[$class] = array_merge($attributeSet, $downstreamSet);
            }

            return ['classToAttribute' => $result, 'mapperTypes' => $mapperTypes];
        });
    }
    
}
