<?php

namespace InSegment\ApiCore\Api\V_2_0;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;

use InSegment\ApiCore\Models\SADSource;
use InSegment\ApiCore\Models\SADAttribute;
use InSegment\ApiCore\Models\SourceAttributeDownstream;

class TransferQueryProcessor
{
    /**
     * Stored rules with attributes dictionary
     *
     * @var array [
     *     string $class => \InSegment\ApiCore\Models\SADAttribute,
     *     ...
     * ] 
     */
    public $classToAttributes = [];
    
    /**
     * Data Transfer Object
     *
     * @var \InSegment\ApiCore\Api\V_2_0\DataTransferObject 
     */
    protected $dto;
    
    /**
     * Input dictionary for currently processed transfer
     *
     * @var \InSegment\ApiCore\Api\V_2_0\DTODictionary
     */
    protected $dictionary;
    
    /**
     * Is currently processed transfer contains multiple entries
     *
     * @var bool 
     */
    protected $multiple;
    
    /**
     * Chunk size
     *
     * @var int
     */
    protected $chunkSize;
    
    /**
     * Model class
     *
     * @var string
     */
    protected $resultClass;
    
    /**
     * Source-Attribute-Downstream instance (non-recursive relation iterator)
     *
     * @var \InSegment\ApiCore\Models\SourceAttributeDownstream
     */
    protected $nonRecursiveIterator;
    
    /**
     * Source-Attribute-Downstream consumer
     *
     * @var \InSegment\ApiCore\Api\V_2_0\TraverseStateManager
     */
    protected $transferStateManager;
    
    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Api\V_2_0\DataTransferObject $dto
     * @param int $mode
     */
    public function __construct(DataTransferObject $dto)
    {
        $this->dto = $dto;
        $this->resultClass = $dto->getResultClass();
        $this->dictionary = $dto->getDictionary();
        $this->multiple = $dto->getIsMultiple();
        $this->chunkSize = $dto->getChunkSize();
        
        $rules = $dto->getRules();
        $classToAttribute = [];
        foreach ($rules['classToAttribute'] as $class => $attributes) {
            $classToAttribute[$class] = new SADAttribute($attributes);
        }
        
        $this->classToAttributes = $classToAttribute;
        $this->nonRecursiveIterator = new SourceAttributeDownstream();
        $mapperHanlders = $this->initMapperHandlers($rules['mapperTypes']);
        $this->transferStateManager = new TraverseStateManager($dto->getData(), $mapperHanlders, $dto->getMode());
    }
    
    /**
     * Process the result
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param array|null $options [
     *     'onProgress' => callable|null,
     *     'onSuccess' => callable|null
     * ]
     * @return $this
     */
    public function process(Builder $query, array $options = [])
    {
        $onProgress = $options['onProgress'] ?? null;
        if (isset($onProgress) && !is_callable($onProgress)) {
            throw new \Exception("'onProgress' callback must be callable");
        }

        $onSuccess = $options['onSuccess'] ?? null;
        if (isset($onSuccess) && !is_callable($onSuccess)) {
            throw new \Exception("'onSuccess' callback must be callable");
        }
        
        $attribute = $this->classToAttributes[$this->resultClass];
        if ($this->multiple) {
            $this->chunkQuery($query, $this->chunkSize, function (EloquentCollection $results) use ($attribute, $onProgress) {
                foreach ($this->dto->getTransformations() as $transformation) {
                    $method = array_shift($transformation);
                    $results = $results->$method(...$transformation);
                }
                
                $this->nonRecursiveIterator->newSource(new SADSource($attribute, $results->values()->all(), [
                    'keyBy' => $this->dto->getKeyBy(),
                    'multiple' => true,
                    'dataAddress' => '-',
                    'class' => $this->resultClass,
                    'relation' => null,
                    'relationName' => null
                ]));
                
                $this->nonRecursiveIterator->iterate($this->transferStateManager);
                
                if (isset($onProgress)) {
                    $onProgress($results, $this->transferStateManager);
                }
            });
        } else {
            $result = $query->first();
            $this->nonRecursiveIterator->newSource(new SADSource($attribute, [$result], [
                'multiple' => false,
                'dataAddress' => '-',
                'class' => $this->resultClass,
                'relation' => null,
                'relationName' => null
            ]));
            
            $this->nonRecursiveIterator->iterate($this->transferStateManager);
            if (isset($onProgress)) {
                $onProgress($result, $this->transferStateManager);
            }
        }
        
        if (isset($onSuccess)) {
            $onSuccess($this->transferStateManager);
        }
        
        return $this;
    }
    
    /**
     * Get result of processing
     * 
     * @return mixed
     */
    public function getResult()
    {
        return $this->transferStateManager->getResult();
    }
    
    /**
     * Chunk the results of the query.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param  int  $count
     * @param  callable  $callback
     * @return bool
     */
    protected function chunkQuery($query, $count, callable $callback)
    {
        $baseQuery = $query->getQuery();
        
        if (empty($baseQuery->orders) && empty($baseQuery->unionOrders)) {
            $model = $query->getModel();
            $query->orderBy($model->getQualifiedKeyName(), 'asc');
        }

        $page = 1;
        $offset = $baseQuery->offset ?: 0;
        $limit = $baseQuery->limit;

        do {
            // We'll execute the query for the given page and get the results. If there are
            // no results we can just break and return from here. When there are results
            // we will call the callback with the current chunk of these results here.
            $pageOffset = ($page - 1) * $count;
            $pageLimit = $limit ? min($count, $limit - $pageOffset) : $count;
            if ($pageLimit == 0) {
                break;
            }
            
            $results = $query->skip($offset + $pageOffset)->take($pageLimit)->get();
            $countResults = $results->count();

            if ($countResults == 0) {
                break;
            }

            // On each chunk result set, we will pass them to the callback and then let the
            // developer take care of everything within the callback, which allows us to
            // keep the memory low for spinning through large result sets for working.
            if ($callback($results, $page) === false) {
                return false;
            }

            unset($results);

            $page++;
        } while ($countResults == $count);

        return true;
    }

    /**
     * Initialise mapper handlers instances with the processor
     * 
     * @param array $mappers
     * @return \InSegment\ApiCore\Api\V_2_0\MappingTypes\MappingType[]
     */
    protected function initMapperHandlers(array $mappers)
    {
        $mapperHandlers = [];
        foreach ($mappers as $type => $handlerClass) {
            $mapperHandlers[$type] = new $handlerClass($this->nonRecursiveIterator, $this);
        }
        
        return $mapperHandlers;
    }
}
