<?php

namespace InSegment\ApiCore\Api\V_2_0;

use InSegment\ApiCore\Api\V_2_0\MappingTypes\MappingType;
use InSegment\ApiCore\Interfaces\DTOCharacteristics;
use InSegment\ApiCore\Facades\DTODefs;

use InSegment\ApiCore\Interfaces\SADConsumer;
use InSegment\ApiCore\Models\TraverseState;

class TraverseStateManager implements SADConsumer
{
    /**
     * DTODefs instance
     *
     * @var \InSegment\ApiCore\Api\V_2_0\DTOUtil
     */
    protected $dtoDefs;
    
    /**
     * Traverse state (mutable shared object)
     *
     * @var \InSegment\ApiCore\Models\TraverseState
     */
    protected $traverseState;
    
    /**
     * Mapper handlers for mapping types
     *
     * @var array
     */
    protected $mapperHandlers;
    
    /**
     * Constructor
     * 
     * @param mixed $data
     * @param array $mapperHandlers
     * @param int $mode
     */
    public function __construct($data, array $mapperHandlers, int $mode)
    {
        $this->dtoDefs = DTODefs::instance();
        $this->mapperHandlers = $mapperHandlers;
        $this->traverseState = new TraverseState($data);
        $this->traverseState->mode = $mode;
    }
    
    public function getResult()
    {
        return $this->traverseState->data;
    }
    
    public function clearResult()
    {
        $this->traverseState->clearResult();
    }
    
    public function sourceStarted(array $params)
    {
        $this->traverseState->stackNewLevel();
        
        $dataAddress = $params['dataAddress'];
        if ($dataAddress !== '-') {
            $this->traverseState->repointData($dataAddress);
        }
        
        if ($params['multiple']) {
            $this->traverseState->initCollection($params['keys'] ?? null);
        }
        
        $this->traverseState->model = null;
        $this->traverseState->mapping = null;
        $this->traverseState->class = $params['class'];
        $this->traverseState->relation = $params['relation'];
        $this->traverseState->relationName = $params['relationName'];
    }

    public function acceptModel($model, array $params)
    {
        $this->traverseState->model = $model;
        
        if ($params['multiple']) {
            if (isset($params['keyBy'])) {
                $this->traverseState->advanceCollection($this->dtoDefs->getResultKeyValues($model, $params['keyBy']));
            } else {
                $this->traverseState->advanceCollection();
            }
        }
    }

    /**
     * 
     * @param array $ruleMapping
     * @param array $param
     */
    public function acceptAttribute($ruleMapping, array $params)
    {
        $this->traverseState->mapping = $ruleMapping;
        
        if ($ruleMapping['dataAddress'] === '-') {
            $this->runHandlerAction($this->mapperHandlers[$ruleMapping['type']]);
        } else {
            $modelLevelData = & $this->traverseState->data;
            $this->traverseState->repointData($ruleMapping['dataAddress']);
            $this->runHandlerAction($this->mapperHandlers[$ruleMapping['type']]);
            $this->traverseState->data = & $modelLevelData;
        }
        
        $this->traverseState->mapping = null;
    }

    public function downstreamStarted(array $params)
    {
        $this->traverseState->stackNewLevel();
    }

    public function downstreamEnded(array $params)
    {
        $this->traverseState->unstackLevel();
    }
    
    public function acceptDownstream($downstream, array $params)
    {
        $closure = $downstream['closure'];
        $this->traverseState->embraceMoment($downstream['moment']);
        $closure($this->traverseState);
    }

    public function saveModel($model, array $params)
    {
        // not implemented yet
    }
    
    public function sourceEnded(array $params)
    {
        $this->traverseState->unstackLevel();
    }

    protected function runHandlerAction(MappingType $handler)
    {
        switch($this->traverseState->mode) {
            case DTOCharacteristics::MODE_IMPORT:
            case DTOCharacteristics::MODE_UPDATE:
                $handler->import($this->traverseState);
            break;
            case DTOCharacteristics::MODE_EXPORT:
                $handler->export($this->traverseState);
            break;
            default: throw new \Exception("TraverseStateManager is in undefined mode!");
        }
    }

}
