<?php

namespace InSegment\ApiCore\Bootstrap;

use InSegment\ApiCore\Providers\ApiCoreServiceProvider;
use InSegment\ApiCore\Providers\EloquentSerivceProvider;

class Bootstrapper
{
    /**
     * Application instance
     * 
     * @var \Illuminate\Contracts\Foundation\Application
     */
    protected $app;
    
    /**
     * Application boostrap dir
     *
     * @var string
     */
    protected $dir;

    /**
     * @var bool
     */
    protected $middlewares = true;

    /**
     * @var bool
     */
    protected $session = true;

    /**
     * @var bool
     */
    protected $routes = true;
    
    /**
     * Constructor
     * 
     * @param \Illuminate\Contracts\Foundation\Application $app
     * @param string $dir
     */
    public function __construct($app, string $dir)
    {
        $this->app = $app;
        $this->dir = $dir;
    }
    
    /**
     * Initialise ApiCore middlewares
     * 
     * @param bool $bool
     * @return $this
     */
    public function withMiddlewares(bool $bool = true)
    {
        $this->middlewares = $bool;
        return $this;
    }
    
    /**
     * Add session support
     * 
     * @param bool $bool
     * @return $this
     */
    public function withSession(bool $bool = true)
    {
        $this->session = $bool;
        return $this;
    }
    
    /**
     * Add transfer api routes
     * 
     * @param bool $bool
     * @return $this
     */
    public function withRoutes(bool $bool = true)
    {
        $this->routes = $bool;
        return $this;
    }
    
    /**
     * Bootstrap ApiCore application
     * 
     * @return null
     */
    public function bootstrap()
    {
        $app = $this->app;
        
        $app->withEloquent();
        $app->configure('api_core');
        
        if ($this->middlewares) {
            $app->middleware([
                \Illuminate\Cookie\Middleware\EncryptCookies::class,
                \Illuminate\Cookie\Middleware\AddQueuedCookiesToResponse::class,
                \Illuminate\Session\Middleware\StartSession::class
            ]);

            $app->routeMiddleware([
                'api.key' => \InSegment\ApiCore\Middleware\VerifyApiKey::class,
                'api.version' => \InSegment\ApiCore\Middleware\ChooseVersion::class,
                'api.gzencrypt' => \InSegment\ApiCore\Middleware\GzipAndEncrypt::class
            ]);
        }
        
        if ($this->session) {
            $app->configure('session');
            $app->singleton('session', function () use ($app) {
                return new \Illuminate\Session\SessionManager($app);
            });

            $app->singleton('session.store', function () use ($app) {
                return $app->make('session')->driver();
            });

            $app->singleton(\Illuminate\Session\Middleware\StartSession::class, function () use ($app) {
                return new \Illuminate\Session\Middleware\StartSession($app->make('session'));
            });

            $app->register(\Illuminate\Cookie\CookieServiceProvider::class);
            $app->bind('Illuminate\Contracts\Cookie\QueueingFactory', 'cookie');
            
        }
        
        $app->register(ApiCoreServiceProvider::class);
        $app->register(EloquentSerivceProvider::class);
        
        if ($this->routes) {
            $prefix = config('api_core.route_prefix');
            
            if (isset($app->router)) {
                // Lumen >= 5.5
                $router = $app->router;
            } else {
                // Lumen <= 5.4
                $router = $app;
            }
            
            $router->group(
                [
                    'prefix' => $prefix,
                    'namespace' => 'InSegment\ApiCore\Controllers',
                ],
                function ($app) {
                    require "{$this->dir}/../vendor/insegment/api-core/routes/lumen/transfer_api.php";
                }
            );
        }
    }
    
}
