<?php

namespace InSegment\ApiCore\Console\Commands;

use Illuminate\Support\Facades\DB;
use Illuminate\Console\Command;

use InSegment\ApiCore\Models\Transaction;
use InSegment\ApiCore\Models\SliceOperation;
use InSegment\ApiCore\Services\Transactor;
use InSegment\ApiCore\Middleware\ChooseVersion;

class GarbageCollectTransactions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'transactions:gc '
                         . '{--v= : Specify API version to use enumeration for, current by default}'
                         . '{--all : Clear all transactions, not just outdated}'
                         . '{--clean : Re-clear clean transactions}'
                         . '{--old : Use old format with session id instead of uid}'
                         . '{--UID= : specify concrete faulty transaction UID to gc}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Removes temporary tables of transactions from crashed or timed out sessions';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $this->info("Checking for garbage transaction tables:");
        $version = $this->option('v') ?: ChooseVersion::getCurrentVersion();
        $all = (bool) $this->option('all');
        $clean = (bool) $this->option('clean');
        $old = (bool) $this->option('old');
        $UID = $this->option('UID') ?? '';
        ChooseVersion::chooseVersion($version);
        
        if ($UID && $all) {
            $clearableSessions = [$UID];
        } else {
            if ($clean) {
                $transactionsQuery = Transaction::query();
                $slicesQuery = SliceOperation::query();
            } else {
                $transactionsQuery = Transaction::where('clean_status', '!=', Transaction::CLEAN_STATUS_CLEAN);
                $slicesQuery = SliceOperation::where('clean_status', '!=', Transaction::CLEAN_STATUS_CLEAN);
            }
            
            if (!$all) {
                $transactionsQuery->outdated();
                $slicesQuery->outdated();
            }
            
            if ($UID) {
                $transactionsQuery->where($old ? 'session_id' : 'uid', '=', $UID);
                $slicesQuery->where('uid', '=', $UID);
            }
            
            $clearableSessions = array_merge(
                $transactionsQuery->pluck($old ? 'session_id' : 'uid')->toArray(),
                $slicesQuery->pluck('uid')->toArray()
            );
        }
        
        $failedUids = [];
        $clearableTables = Transactor::tablesForGC($clearableSessions, $old);
        $database = Transactor::getTransactionSchema();
        foreach ($clearableTables as $uid => $tables) {
            foreach ($tables as $table) {
                try {
                    $result = DB::statement("DROP TABLE IF EXISTS `{$database}`.`{$table}`");
                } catch (\Throwable $e) {
                    $result = false;
                }

                if ($result) {
                    $this->info("DROP: {$table}");
                } else {
                    $failedUids[] = $uid;
                    $this->info("Failed to DROP: {$table}");
                }
            }
        }
        
        $successUids = array_diff($clearableSessions, $failedUids);
        if (count($successUids)) {
            Transaction::whereIn($old ? 'session_id' : 'uid', $successUids)->update(['clean_status' => Transaction::CLEAN_STATUS_CLEAN]);
            SliceOperation::whereIn('uid', $successUids)->update(['clean_status' => Transaction::CLEAN_STATUS_CLEAN]);
        }
        
        $this->info("done");
    }
}
