<?php

namespace InSegment\ApiCore\Exceptions;

use InSegment\ApiCore\Services\ExceptionDebug;

class ApiTransactorException extends \Exception
{
    const CODE_UNKNOWN = 1000;
    const CODE_WRONG_SEQUENCE = 1001;
    const CODE_FAIL_UNDONE = 1002;
    const CODE_FAIL_NOT_UNDONE = 1003;
    const CODE_WRONG_WEIGHT = 1004;
    const CODE_IMPOSSIBLE_INSERT = 1005;
    const CODE_IMPOSSIBLE_UPDATE = 1006;
    const CODE_WRONG_TABLE_FOR_SAVE = 1007;
    const CODE_UNSUPPORTED_BUFFER_TYPE = 1008;
    const CODE_SERVER_FAULT = 1009;
    const CODE_COUNT_MISSING = 1010;
    const CODE_COUNT_INVALID = 1011;
    const CODE_CREATE_TABLE_FAIL = 1012;
    const CODE_ALTER_TABLE_FAIL = 1013;
    const CODE_UNEXPECTED_CASUALTIES = 1014;
    const CODE_UNEXPECTED_CLASS = 1015;
    const CODE_POST_PROCESS_FAILED = 1016;
    const CODE_RECURSIVE_TRANSACTION = 1017;
    const CODE_TABLE_NOT_INCLUDED = 1018;
    const CODE_UNABLE_TO_GENERATE_UID = 1019;
    const CODE_NO_DEFAULT_MEMORY_ENGINE_LIMIT = 1020;
    const CODE_UNKNOWN_TYPE_OF_MOVE_OPERATION = 1021;
    const CODE_AFFECTED_LESS_GENERATIONS_THAN_INSERTED = 1022;
    const CODE_UUIDABLES_WRONG_ORDER = 1023;
    const CODE_CHANGES_NEED_PRIMARY_KEY = 1024;

    private $data;

    public function __construct(int $code = self::CODE_UNKNOWN, $previous = null) {
        parent::__construct("", $code, $previous);
    }

    public function setData($data)
    {
        $this->data = $data;
        return $this;
    }

    public function getData()
    {
        if (($previous = $this->getPrevious())) {
            ExceptionDebug::makeDebug($this->data, $previous, 'previous');
        }

        return $this->data;
    }

    public function compile(...$args)
    {
        self::normalizeArgs($args);
        switch($this->code) {
            case self::CODE_UUIDABLES_WRONG_ORDER:
                $this->message = "The column(s) {$args[0]} of table {$args[1]} depend on auto-key of table {$args[2]} which is not yet merged! Need to change the merge order of these tables in uuidables.json!";
            break;
            case self::CODE_UNEXPECTED_CASUALTIES:
                $this->message = "When moving objects ({$args[0]}) for table '{$args[1]}', affected rows ({$args[2]}) for operation (SQL: {$args[3]}) were different from estimate ({$args[4]})!";
            break;
            case self::CODE_IMPOSSIBLE_INSERT:
                $this->message = "Cannot use index for object of class {$args[0]} in insert table < update separator: {$args[1]} < {$args[2]}";
            break;
            case self::CODE_IMPOSSIBLE_UPDATE:
                $this->message = "Cannot use index for object of class {$args[0]} in update table >= update separator: {$args[1]} >= {$args[2]}";
            break;
            case self::CODE_ALTER_TABLE_FAIL:
                $this->message = "Failed to {$args[0]} of {$args[1]} table for original table: {$args[2]}";
            break;
            case self::CODE_UNABLE_TO_GENERATE_UID:
                $this->message = "Failed to generate UID of transaction. Transaction cannot be open";
            break;
            case self::CODE_CHANGES_NEED_PRIMARY_KEY:
                $this->message = "To create changes temporary table for {$args[0]} it needs to have a primary key!";
            break;
            case self::CODE_CREATE_TABLE_FAIL:
                $this->message = "Failed to create temporary table for {$args[0]} into {$args[1]}";
            break;
            case self::CODE_WRONG_TABLE_FOR_SAVE:
                $this->message = "Niether insert or update table is set. Table set: {$args[0]}";
            break;
            case self::CODE_FAIL_UNDONE:
                $this->message = "Failed to perform {$args[0]} operation on table {$args[1]}";
            break;
            case self::CODE_TABLE_NOT_INCLUDED:
                $this->message = "The table {$args[0]} is not included in transaction";
            break;
            case self::CODE_UNSUPPORTED_BUFFER_TYPE:
                $this->message = "Do not know how to make buffer for: {$args[0]}";
            break;
            case self::CODE_UNKNOWN_TYPE_OF_MOVE_OPERATION:
                $this->message = "Unknown type of move operation: {$args[0]}";
            break;
            case self::CODE_COUNT_MISSING:
                $this->message = "Missing count for table: {$args[0]}";
            break;
            case self::CODE_COUNT_INVALID:
                $this->message = "Invalid count for table: {$args[0]}";
            break;
            case self::CODE_UNEXPECTED_CLASS:
                $this->message = "The Transactor do not expect {$args[0]}";
            break;
            case self::CODE_WRONG_SEQUENCE:
                $this->message = "Transaction is {$args[0]}";
            break;
            case self::CODE_FAIL_NOT_UNDONE:
                $this->message = 'Transaction is abnormal!!!';
            break;
            case self::CODE_POST_PROCESS_FAILED:
                $this->message = 'Post-processing of transaction failed!';
            break;
            case self::CODE_RECURSIVE_TRANSACTION:
                $this->message = 'Transactor table-watching entrypoint was accessed twice; it cannot be recursed';
            break;
            case self::CODE_NO_DEFAULT_MEMORY_ENGINE_LIMIT:
                $this->message = 'No default MEMORY engine limit specified in DTODefs!';
            break;
            case self::CODE_AFFECTED_LESS_GENERATIONS_THAN_INSERTED:
                $this->message = "For table {$args[0]} affected less generations ({$args[1]}) than expected as moved by insert ({$args[2]})!";
            break;
            
            case self::CODE_WRONG_WEIGHT:
            case self::CODE_SERVER_FAULT:
                $this->message = $args[0];
            break;
        }
        return $this;
    }

    protected static function normalizeArgs(&$args)
    {
        foreach(range(0, 4) as $i) {
            if (!array_key_exists($i, $args)) {
                $args[$i] = '{UNFILLED PLACEHOLDER}';
            }
        }
    }
}
