<?php

namespace InSegment\ApiCore\Exceptions;

use Throwable;
use Illuminate\Auth\AuthenticationException;
use InSegment\ApiCore\Services\Transactor;
use InSegment\ApiCore\Services\ExceptionDebug;
use Symfony\Component\Console\Output\OutputInterface;

define("ERR", "error");
define("CODE", "code");
define("DATA", "data");
define("MSG", "message");
define("FILE", "file");
define("LINE", "line");
define("SQL", "sql");

trait BaseHandling
{
    /**
     * Report or log an exception.
     *
     * This is a great spot to send exceptions to Sentry, Bugsnag, etc.
     *
     * @param  \Throwable  $exception
     * @return void
     */
    public function report(Throwable $exception) {
        parent::report($exception);
    }

    /**
    * Render an exception into an HTTP response.
    *
    * @param  \Illuminate\Http\Request  $request
    * @param  \Throwable  $exception
    * @return \Illuminate\Http\Response
    */

    public function render($request, Throwable $exception) {
        //check if exception is an instance of ModelNotFoundException or NotFoundHttpException
        // if($exception instanceof ModelNotFoundException or $exception instanceof NotFoundHttpException) {
        //     // ajax 404 json feedback
        //     if($request->ajax()) {
        //         return response()->json([MSG => "Not Found"], 404);
        //     }
        //     // normal 404 view page feedback
        //     return response()->view('errors.404', [], 404);
        // }

        // if the exception happened during transaction, wrap it in transactor's exception
        $exception = Transactor::affectOnYourWatch($exception);

        // TODO: add middleware instead
        $isApiException = $request->isJson() && $exception instanceof ApiTransactorException;

        if($exception instanceof NotFoundHttpException) {
            // dd("NotFoundHttpException");
        }
        if($exception instanceof HttpResponseException) {
            // dd("HttpResponseException");
            return $exception->getResponse();
        } elseif ($exception instanceof ModelNotFoundException) {
            // dd("ModelNotFoundException");
            $exception = new NotFoundHttpException($exception->getMessage(), $exception);
        } elseif ($exception instanceof AuthenticationException) {
            // HTTP исключение при утрате токена или истекшей сессии
            return $this->unauthenticated($request, $exception);
        } elseif ($exception instanceof AuthorizationException) {
            // dd("AuthorizationException");
            $exception = new HttpException(403, $e->getMessage());
        } elseif ($exception instanceof ValidationException && $exception->getResponse()) {
            // dd("ValidationException");
            return $exception->getResponse();
        }

        // если запрос типа ajax вернуть REST ответ
        if($request->isXmlHttpRequest() || $request->expectsJson() || $isApiException) {
            $response[ERR] = 400;
            $response[MSG] = "SERVER_ERROR";

            ExceptionDebug::makeDebug($response, $exception);

            if($exception instanceof \Illuminate\Auth\Access\AuthorizationException) {
                $response[ERR] = 403;
                $response[MSG] = "Action is unauthorized";
            }
            // Ошибка запрещения доступа
            if($exception instanceof \Symfony\Component\HttpKernel\Exception\UnauthorizedHttpException) {
                $response[ERR] = 403;
                $response[MSG] = "ACCESS DENIED";
            }
            // Ошибки транзактора
            if($exception instanceof ApiTransactorException) {
                $response[MSG] = "API_TRANSACTOR_ERROR";
            }
            // SQL исключения
            if($exception instanceof \PDOException) {
                $response[MSG] = "SQL_ERROR";
            }
            return response()->json($response, $response[ERR]);
        }
        return parent::render($request, $exception);
    }
    
    /**
     * Render an exception to the console.
     *
     * @param  \Symfony\Component\Console\Output\OutputInterface  $output
     * @param  \Throwable  $e
     * @return void
     */
    public function renderForConsole($output, Throwable $e)
    {
        if (config("app.debug")) {
            $output->setVerbosity(OutputInterface::VERBOSITY_VERY_VERBOSE);
        }

        parent::renderForConsole($output, $e);
    }
    
    /**
     * Convert an authentication exception into an unauthenticated response.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Illuminate\Auth\AuthenticationException  $exception
     * @return \Illuminate\Http\Response
     */
    protected function unauthenticated($request, AuthenticationException $exception)
    {
        if ($request->expectsJson()) {
            return response()->json(['error' => 'Unauthenticated.'], 401);
        }

        return redirect()->guest(route('login'));
    }
}
