<?php

namespace InSegment\ApiCore\Inspections;

use Illuminate\Support\Arr;
use InSegment\ApiCore\Api\StaticRuleAnalysis;
use InSegment\ApiCore\Interfaces\StaticRuleInspection;

class AutoCountInspection implements StaticRuleInspection
{
    /**
     * StaticRuleAnalysis instance
     * 
     * @var \InSegment\ApiCore\Api\StaticRuleAnalysis 
     */
    protected $analyser;
    
    /**
     * Extra data for inspection
     * 
     * @var array
     */
    protected $inspectData;
    
    /**
     * Output for the auto counts calculated
     * May also contain counts supplied by client
     * 
     * @var array 
     */
    protected $countOutput;
    
    /**
     * The properties below are set on the inspection iteration
     */
    
    /**
     * dotFormat of the rule leading to currentMapping during the course of its inspection
     * 
     * @var string 
     */
    protected $dotFormat;
    
    /**
     * The mapping currently being inspected during the course of its inspection
     * 
     * @var array 
     */
    protected $currentMapping;
    
    /**
     * Current parent Model class during the course of concrete mapping inspection
     * 
     * @var string 
     */
    protected $parentClass;
    
    /**
     * Current related Model class during the course of concrete mapping inspection
     * 
     * @var string 
     */
    protected $relatedClass;
    
    /**
     * Table of the Model of parent class type during the course of concrete mapping inspection
     *
     * @var string
     */
    protected $parentTable;
    
    /**
     * Table of the Model of related class type during the course of concrete mapping inspection
     *
     * @var string
     */
    protected $currentTable;
    
    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Api\StaticRuleAnalysis $analyser
     * @param array $countOutput
     * @param array $autoSetTables
     */
    public function __construct(StaticRuleAnalysis $analyser, array &$countOutput, array &$inspectData)
    {
        $this->analyser = $analyser;
        $this->countOutput = &$countOutput;
        $this->inspectData = &$inspectData;
    }
    
    /**
     * Getter for StaticRuleAnalysis instance
     * 
     * @return StaticRuleAnalysis
     */
    public function getAnalyser(): StaticRuleAnalysis
    {
        return $this->analyser;
    }

    /**
     * Main inspection method
     * Accepts information from analyser, returns confirmation to proceed further with this inspection
     * 
     * @param string $dotFormat
     * @param array $mapping
     * @param string $handlerClass
     * @param string $relatedClass
     * @return bool
     */
    public function inspectRule($dotFormat, &$mapping, $handlerClass, $relatedClass): bool
    {
        if (!$handlerClass || !$relatedClass) {
            return false;
        }
        
        $this->dotFormat = $dotFormat;
        $this->currentMapping = &$mapping;
        $this->parentClass = $this->getAnalyser()->getParentClass();
        $this->relatedClass = $relatedClass;
        $this->parentTable = (new $this->parentClass)->getTable();
        $this->currentTable = (new $relatedClass)->getTable();
        
        if ($this->doesNeedDynamicUnderweight()) {
            $this->inspectData['dynUW'][$this->currentTable] = true;
        }

        if ($this->isAutoCountAllowed($handlerClass)) {
            $this->autoSetCurrentCount();
        } else if ($this->isCurrentCountAuto()) {
            // do not allow auto count on tables with mixed types of join
            $this->clearCurrentAutoCount();
        }
        
        if ($dotFormat === '-' && $this->hasCurrentCount() && $this->isParentCountAuto()) {
            if ($this->countOutput[$this->currentTable] == 0) {
                $this->countOutput[$this->parentTable] = 0;
            }
        }
        
        if ($this->hasCurrentCount()) {
            $this->autoSetAddWith();
        }
        
        return $handlerClass::importsRelation($mapping);
    }
    
    /**
     * Set count automatically for the current table (usually based on parent count)
     * 
     * @return $this
     */
    protected function autoSetCurrentCount()
    {
        if ($this->hasParentCount()) {
            if (!$this->hasCurrentCount()) {
                $this->countOutput[$this->currentTable] = $this->getAutoAmount();
                $this->inspectData['auto'][$this->currentTable] = true;
            } else if (!empty($this->inspectData['auto'][$this->currentTable])) {
                $this->countOutput[$this->currentTable] += $this->getAutoAmount();
            }
        }
        
        return $this;
    }
    
    /**
     * Respect 'addWith' mapping construction
     * 
     * @return $this
     */
    protected function autoSetAddWith()
    {
        if (isset($this->currentMapping['addWith'])) {
            foreach ($this->currentMapping['addWith'] as $class => $addWith) {
                if (is_int($class)) {
                    $table = (new $addWith)->getTable();
                    $addWith = 1;
                } else {
                    $table = (new $class)->getTable();
                }
                
                $data = Arr::wrap($addWith);
                $amount = $data[0] * $this->countOutput[$this->currentTable];
                $dynUW = !isset($this->inspectData['dynUW'][$table]) && in_array('dynUW', $addWith);
                $dynUW && ($this->inspectData['dynUW'][$table] = true);
                
                // only set if the count output for $table is auto or not set (makes auto)
                if (!isset($this->countOutput[$table])) {
                    $this->countOutput[$table] = $amount;
                    $this->inspectData['auto'][$table] = true;
                } else if (!empty($this->inspectData['auto'][$table])) {
                    $this->countOutput[$table] += $amount;
                }
            }
        }
        
        return $this;
    }
    
    /**
     * Invalidate previously set auto count for the current table
     * 
     * @return $this
     */
    protected function clearCurrentAutoCount()
    {
        unset(
            $this->inspectData['auto'][$this->currentTable],
            $this->inspectData['dynUW'][$this->currentTable],
            $this->countOutput[$this->currentTable]
        );
        
        return $this;
    }
    
    /**
     * Get count amount to auto set
     * 
     * @return int
     */
    protected function getAutoAmount()
    {
        return $this->countOutput[$this->parentTable];
    }

    /**
     * Whether count for the parent table is already present
     * 
     * @return bool
     */
    protected function hasParentCount(): bool
    {
        return isset($this->countOutput[$this->parentTable]);
    }
    
    /**
     * Whether count for the current table is already present
     * 
     * @return bool
     */
    protected function hasCurrentCount(): bool
    {
        return isset($this->countOutput[$this->currentTable]);
    }
    
    /**
     * Whether the current mapping suits for auto counting (usually one-to-one mappings)
     * 
     * @param string $handlerClass
     * @return bool
     */
    protected function isAutoCountAllowed(string $handlerClass): bool
    {
        return $handlerClass::isAutoCount($this->currentMapping);
    }
    
    /**
     * Whether count for current table is already automatic
     * 
     * @return bool
     */
    protected function isParentCountAuto(): bool
    {
        return !empty($this->inspectData['auto'][$this->parentTable]);
    }

    /**
     * Whether count for current table is already automatic
     * 
     * @return bool
     */
    protected function isCurrentCountAuto(): bool
    {
        return !empty($this->inspectData['auto'][$this->currentTable]);
    }
    
    /**
     * Return whether the dynUW for the current mapping should be set
     * 
     * @return bool
     */
    protected function doesNeedDynamicUnderweight(): bool
    {
        return !isset($this->inspectData['dynUW'][$this->currentTable])
            && in_array('dynUW', Arr::get($this->currentMapping, 'extra', []));
    }

}
