<?php

namespace InSegment\ApiCore\Inspections;

use Illuminate\Support\Arr;
use InSegment\ApiCore\Facades\DTODefs;
use InSegment\ApiCore\Services\Transactor;
use InSegment\ApiCore\Api\StaticRuleAnalysis;
use InSegment\ApiCore\Interfaces\StaticRuleInspection;

/**
 * Inspection that gets retrieve closures from relation structure
 * for dictionaries and ahead-in-transaction supplied models
 */
class PreloadInspection implements StaticRuleInspection
{
    /**
     * DTODefs instance
     * 
     * @var \InSegment\ApiCore\Facades\DTODefs 
     */
    protected $dtoDefs;
    
    /**
     * StaticRuleAnalysis instance
     * 
     * @var \InSegment\ApiCore\Api\StaticRuleAnalysis 
     */
    protected $analyser;
    
    /**
     * Output of inspection
     * 
     * @var array 
     */
    protected $out;
    
    /**
     * Keys of related Models tables
     * 
     * @var array
     */
    protected $relatedKeys;
    
    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Api\StaticRuleAnalysis $analyser
     * @param array $out
     */
    public function __construct(StaticRuleAnalysis $analyser, array &$out)
    {
        $this->out = &$out;
        $this->dtoDefs = DTODefs::instance();
        $this->analyser = $analyser;
        $this->relatedKeys = [];
    }
    
    /**
     * Get analyzer
     * 
     * @return StaticRuleAnalysis
     */
    public function getAnalyser(): StaticRuleAnalysis
    {
        return $this->analyser;
    }

    /**
     * Main inspection method
     * Accepts information from analyser, returns confirmation to proceed further with this inspection
     * 
     * @param string $dotFormat
     * @param array $mapping
     * @param string $handlerClass
     * @param string $relatedClass
     * @return bool
     */
    public function inspectRule($dotFormat, &$mapping, $handlerClass, $relatedClass): bool {
        if ($handlerClass && $relatedClass) {
            $preloadOptions = $this->getPreloadOptions($mapping);

            // 'ahead' loads entries sent earlier in the same transaction
            // 'dictionary' loads all entries from some table
            // 'wildcard' uses referenceWidcard from mapping to get a set if identifiers by the address in the input
            $meetOptions = $preloadOptions['ahead'] || $preloadOptions['dictionary'] || $preloadOptions['wildcard'];
            $parentClass = $this->getAnalyser()->getParentClass();
            if ($meetOptions && ($relationName = $handlerClass::getRelationName($parentClass, $mapping))) {
                $extKeysInfo = $handlerClass::getExtKeysInfo($this->dtoDefs, $mapping);
                $extKeys = array_keys($extKeysInfo);
                $extKeysImplode = implode(',', $extKeys);
                
                // keep reference to keys for the current class
                $this->preparePreloadKeys($mapping, $handlerClass, $relatedClass, $parentClass, $relationName, $extKeysInfo, $extKeysImplode);
                $this->out['suppliers'][$parentClass][$relatedClass][$extKeysImplode] = $this->getResultSupplier($preloadOptions, $parentClass, $relatedClass, $extKeys, $extKeysImplode);
            }
        }

        return $handlerClass && $relatedClass;
    }
    
    /**
     * Get result supplier
     * 
     * @param array $preloadOptions
     * @param string $parentClass
     * @param string $relatedClass
     * @param array $extKeys
     * @param string $extKeyImplode
     * @return \Closure
     */
    private function getResultSupplier(&$preloadOptions, $parentClass, $relatedClass, &$extKeys, $extKeyImplode)
    {
        $result = null; $resultSet = false;
        $querySupplier = $this->getQuerySupplier($preloadOptions, $parentClass, $relatedClass, $extKeys, $extKeyImplode);
        
        // only use reference to keys when the query is actually done - that way, it certainly will
        // happen after all possibly used keys in the dictionary are inspected
        return function($affector = null) use (&$result, &$resultSet, $relatedClass, &$extKeys, $extKeyImplode, $querySupplier) {
            if ($resultSet) {
                return $result;
            }
            
            if (isset($affector)) {
                $query = ($query = $querySupplier()) ? ($affector($query) ?? $query) : $query;
            } else {
                $query = $querySupplier();
            }
            
            if ($query) {
                $result = $query->get($this->relatedKeys[$relatedClass][$extKeyImplode])->groupBy(function ($item) use (&$extKeys) {
                    $group = [];
                    foreach ($extKeys as $key) {
                        $group[$key] = $item->$key;
                    }
                    
                    return implode(',', $group);
                });
            }
            
            $resultSet = true;
            return $result;
        };
    }
    
    /**
     * Get query supplier for preloading of related class
     * 
     * @param array $preloadOptions
     * @param string $parentClass
     * @param string $relatedClass
     * @param array $extKeys
     * @param string $extKeyImplode
     * @return \Closure
     */
    private function getQuerySupplier(&$preloadOptions, $parentClass, $relatedClass, &$extKeys, $extKeyImplode)
    {
        if ($preloadOptions['ahead']) {
            return function () use ($relatedClass) {
                return Transactor::receivedModelsScope($relatedClass::query());
            };
        } else if ($preloadOptions['wildcard']) {
            if (!isset($this->out['referenceWildcards'][$parentClass][$relatedClass][$extKeyImplode])) {
                $this->out['referenceWildcards'][$parentClass][$relatedClass][$extKeyImplode] = [];
            }

            if (!in_array($preloadOptions['wildcard'], $this->out['referenceWildcards'][$parentClass][$relatedClass][$extKeyImplode])) {
                $this->out['referenceWildcards'][$parentClass][$relatedClass][$extKeyImplode][] = $preloadOptions['wildcard'];
            }

            $this->out['reference'][$parentClass][$relatedClass][$extKeyImplode] = [];
            $reference = &$this->out['reference'][$parentClass][$relatedClass][$extKeyImplode];

            // add whereIn using refernce to $reference, which is a referrence to $this->out['reference'][...][...]

            if (count($extKeys) > 1) {
                return function () use ($relatedClass, &$extKeys, &$reference) {
                    $keys = '(`' . implode('`,`', $extKeys) . '`)';
                    $combinations = array_unique($reference, SORT_REGULAR);
                    $oneFiller = implode(',', array_fill(0, count($extKeys), '?'));
                    $placeholders = '(' . implode(',', array_fill(0, count($combinations), "({$oneFiller})")) . ')';
                    $bindings = Arr::flatten($combinations);
                    return $relatedClass::query()->whereRaw("{$keys} IN {$placeholders}", $bindings);
                };
            } else {
                return function () use ($relatedClass, $extKeyImplode, &$reference) {
                    return $relatedClass::query()->whereIn($extKeyImplode, array_values(array_unique($reference)));
                };
            }
        } else {
            return function () use ($relatedClass) {
                return $relatedClass::query();
            };
        }
    }

    /**
     * Join together keys need for preloading and give a reference
     * 
     * @param array $mapping
     * @param string $handlerClass
     * @param string $relatedClass
     * @param string $parentClass
     * @param string $relationName
     * @param array $extKeysInfo
     * @param string $extKeyImplode
     * @return null
     */
    private function preparePreloadKeys($mapping, $handlerClass, $relatedClass, $parentClass, $relationName, $extKeysInfo, $extKeyImplode)
    {
        if ($handlerClass::importsRelation($mapping)) {
            // if the model happens to be importable, need to select all columns
            // this is unlikely to happen, because why preload importable models? better to eager load them
            $this->relatedKeys[$relatedClass][$extKeyImplode] = ['*'];
        } else if (($this->relatedKeys[$relatedClass][$extKeyImplode] ?? null) !== ['*']) {
            $classRules = &$this->getAnalyser()->getRules($relatedClass);
            
            // keys to load on the preload: merge previously computed ones, plus unuqie ones from relation constraints
            // and plus for uniques using the key specified in mapping
            $this->relatedKeys[$relatedClass][$extKeyImplode] = array_unique(array_merge(
                $this->relatedKeys[$relatedClass][$extKeyImplode] ?? [],
                array_keys((new $parentClass)->$relationName()->constraintsToAttributes(new $relatedClass)),
                $handlerClass::getSearchAttributeKeys($this->dtoDefs, $mapping, $classRules, $relatedClass, $extKeysInfo)
            ));
        }
    }
    
    /**
     * Return preload options from mapping
     * 
     * @param array $mapping
     * @return array
     */
    private function getPreloadOptions(&$mapping)
    {
        $extras = array_fill_keys(Arr::get($mapping, 'extra', []), true);
        
        return $extras + ['ahead' => false, 'dictionary' => false, 'wildcard' => Arr::get($mapping, 'referenceWidcard')];
    }

}
