<?php

namespace InSegment\ApiCore\Interfaces;

interface SADConsumer
{
    /**
     * Before the first model of the source is iterated (if the source contains any) this method informs SADConsumer
     * about the start of the source.
     * 
     * @param array $params
     */
    public function sourceStarted(array $params);
    
    /**
     * Before attributes and downstream attributes are processed, this method will be called by SAD instance to
     * start processing of Model. By default, SAD instance will only process models from the initial array -  to add
     * new models, the consumer should call method SourceAttributeDownstream::newSource, which will affect the cycle
     * 
     * @param mixed $model
     * @param array $params
     */
    public function acceptModel($model, array $params);

    /**
     * In the first part of the cycle, SAD instance will process attributes of the models and call this method on each
     * 
     * @param mixed $attribute
     * @param array $params
     */
    public function acceptAttribute($attribute, array $params);
    
    /**
     * After attributes, but before the model downstream is iterated (if there is any)
     * this method informs SADConsumer about the start of downstream.
     * 
     * @param array $params
     */
    public function downstreamStarted(array $params);
    
    /**
     * In the second part of the cycle, SAD instance will process separately downstream attributes of the models and call
     * this method on each. By default, there are none downstream attributes - the consumer should call
     * methods SourceAttributeDownstream::addDownStream to affect the cycle
     * 
     * @param mixed $downstream
     * @param array $params
     */
    public function acceptDownstream($downstream, array $params);
    
    /**
     * After model downstream was iterated (if the there was any)
     * this method informs SADConsumer about the end of downstream.
     */
    public function downstreamEnded(array $params);
    
    /**
     * After attributes and downstream attributes are processed, this method will be called by SAD instance to
     * start processing of Model. By default, SAD instance will only process models from the initial array -  to add
     * new models, the consumer should call method SourceAttributeDownstream::newSource, which will affect the cycle
     * 
     * @param mixed $model
     * @param array $params
     */
    public function saveModel($model, array $params);
    
    /**
     * When the last model of the source iteration finished, this method informs SADConsumer
     * about the end of the source.
     * 
     * If returned value is strictly equal to false, the SourceAttributeDownstream will exit the loop of iteration
     * 
     * @param array $params
     * @return bool|null
     */
    public function sourceEnded(array $params);
}
