<?php

namespace InSegment\ApiCore\Middleware;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\File;
use Closure;

class ChooseVersion
{
    const LEAST_SUPPORTED = 'V_1_0';
    const VERSION_REGEX = '/^V?([0-9]+[._]?)+[0-9a-z]?$/';
    
    /**
     * Handle an incoming request.
     *
     * @param \Illuminate\Http\Request $request
     * @param \Closure  $next
     * @param string $versionKeyParam
     * @param string $namespace
     * @return mixed
     */
    public function handle($request, Closure $next, $versionKeyParam = 'version')
    {
        $route = $request->route();
        $version = (is_array($route) ? Arr::get($route, "2.{$versionKeyParam}") : $route->parameter($versionKeyParam))
            ?: $request->get($versionKeyParam);
        
        self::chooseVersion($version);
        
        return $next($request);
    }
    
    /**
     * Choose default version of the DTODefs
     * 
     * @param string|null $version
     * @param string $namespace
     * @return null
     */
    public static function chooseVersion($version = null)
    {
        $namespace = config('api_core.versions_namespace');
        $folder = self::requireVersionSupport($version ?: self::getCurrentVersion());
        $class = "{$namespace}\\{$folder}\\DTODefs";
        App::instance('DTODefs', new $class);
    }

    /**
     * Get all possible versions
     * 
     * @return array
     */
    public static function getAllVersions(): array
    {
        return array_map('basename', File::directories(self::getVersionsPath()));
    }
    
    /**
     * Get current (usually latest) version
     * 
     * @return string
     */
    public static function getCurrentVersion(): string
    {
        return config('api_core.current_version', self::LEAST_SUPPORTED);
    }
    
    /**
     * Path to the folder of API versions
     * 
     * @return string
     */
    public static function getVersionsPath(): string
    {
        return config('api_core.versions_path');
    }
    
    /**
     * Get folder name for version from version string
     * (replaces dots with underscores and prepends with V_)
     * 
     * @param string $version
     * @return string
     */
    private static function getVersionFolderName(string $version): string
    {
        if (strpos($version, 'V_') !== 0) {
            $version = 'V_' . str_replace('.', '_', $version);
        } else if (strpos($version, '.') !== false) {
            $version = str_replace('.', '_', $version);
        }
        
        return $version;
    }
    
    /**
     * Checks the version string is in acceptable format
     * 
     * @param string $version
     * @return null
     * @throws \Exception
     */
    private static function filterVersion(string $version)
    {
        if (!preg_match(static::VERSION_REGEX, $version)) {
            throw new \Exception("Illegal version format");
        }
    }
    
    /**
     * Require support for the specified version or throw and Exception if no
     * 
     * @param string $version
     * @return string
     * @throws \Excpetion
     */
    private static function requireVersionSupport(string $version): string
    {
        $versionFolder = self::getVersionFolderName($version);
        $list = self::getAllVersions();
        
        if (!in_array($versionFolder, $list)) {
            throw new \Exception("Version {$version} is not a supported version");
        }
        
        return $versionFolder;
    }
}
