<?php

namespace InSegment\ApiCore\Middleware;

use Closure;

use Illuminate\Support\Arr;
use InSegment\ApiCore\Interfaces\ApiKeyInterface;

use Illuminate\Encryption\Encrypter;
use Symfony\Component\HttpFoundation\ParameterBag;

class GzipAndEncrypt
{
    /**
     * The additional attributes passed to the middleware.
     *
     * @var array
     */
    protected $attributes = [];

    /**
     * Current request
     * 
     * @var \Illuminate\Http\Request 
     */
    private $request;
    
    /**
     * Handle an incoming request.
     *
     * @param \Illuminate\Http\Request $request
     * @param \Closure  $next
     * @return mixed
     */
    public function handle($request, Closure $next, ...$attributes)
    {
        $this->request = $request;
        $this->attributes = $attributes;
        $this->clean($request);
        with($response = $next($request))->setContent($this->encryptData($response->getContent()));
        return $response;
    }
    
    /**
     * gzdecode and decrypt data
     *
     * @param  string  $key
     * @param  mixed  $value
     * @return mixed
     */
    protected function transform($key, $value)
    {
        /*if ($this->isGzEncoded($value)) {
            $value = gzdecode($value);
        }*/
        
        if (config('api_core.transaction_use_encryption')) {
            $route = $this->request->route();
            $apiKey = is_array($route) ? Arr::get($route, '2.apiKey') : $route->parameter('apiKey');
            
            if ($apiKey instanceof ApiKeyInterface) {
                $encrypter = new Encrypter($apiKey->getPrivateKey(), config('api_core.cipher', config('app.cipher')));
                return json_decode($encrypter->decrypt($value), true);
            }
        }
        
        return $value;
    }

    /**
     * Checks if the data was gzencoded
     * 
     * @param string $data
     * @return bool
     */
    private function isGzEncoded(&$data)
    {
        return 0 === mb_strpos($data , "\x1f" . "\x8b" . "\x08");
    }

    /**
     * Encrypt Data
     *
     * @param array|string $data Data to crypt
     * @return string
     */
    public function encryptData($data)
    {
        /*if ($this->acceptsGzip($this->request)) {
            $data = gzencode($data);
        }*/
        
        if (config('api_core.transaction_use_encryption')) {
            $route = $this->request->route();
            $apiKey = is_array($route) ? Arr::get($route, '2.apiKey') : $route->parameter('apiKey');
            
            if ($apiKey instanceof ApiKeyInterface) {
                $encrypter = new Encrypter($apiKey->getPrivateKey(), config('app.cipher'));
                return $encrypter->encrypt(json_encode($data));
            }
        }
        
        return $data;
    }

    /**
     * Check if request accepts gzip
     * 
     * @param \Illuminate\Http\Request $request
     * @return boolean
     */
    private function acceptsGzip($request)
    {
        $acceptEncoding = explode(',', $request->headers->get('Accept-Encoding', ''));
        
        foreach ($acceptEncoding as $encoding) {
            if (trim($encoding) == 'gzip') {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Clean the request's data.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return void
     */
    protected function clean($request)
    {
        $this->cleanParameterBag($request->query);

        if ($request->isJson()) {
            $this->cleanParameterBag($request->json());
        } else {
            $this->cleanParameterBag($request->request);
        }
    }

    /**
     * Clean the data in the parameter bag.
     *
     * @param  \Symfony\Component\HttpFoundation\ParameterBag  $bag
     * @return void
     */
    protected function cleanParameterBag(ParameterBag $bag)
    {
        $bag->replace($this->cleanArray($bag->all()));
    }

    /**
     * Clean the data in the given array.
     *
     * @param  array  $data
     * @return array
     */
    protected function cleanArray(array $data)
    {
        return collect($data)->map(function ($value, $key) {
            return $this->cleanValue($key, $value);
        })->all();
    }

    /**
     * Clean the given value.
     *
     * @param  string  $key
     * @param  mixed  $value
     * @return mixed
     */
    protected function cleanValue($key, $value)
    {
        if (is_array($value)) {
            return $this->cleanArray($value);
        }

        return $this->transform($key, $value);
    }

}
