<?php

namespace InSegment\ApiCore\Middleware;

use Closure;

use Illuminate\Http\Request;
use InSegment\ApiCore\Interfaces\ApiKeyInterface;
use Symfony\Component\HttpKernel\Exception\UnauthorizedHttpException;

class VerifyApiKey
{
    private const API_KEY_HEADER_DEFAULT = 'api-key';

    /**
     * Handle an incoming request.
     *
     * @param \Illuminate\Http\Request $request
     * @param \Closure  $next
     * @param string $apiKeyParam
     * @param string $authTypes
     * @return mixed
     */
    public function handle($request, Closure $next, $apiKeyParam, ...$authTypes)
    {
        $publicKey = null;

        if (in_array('header', $authTypes)) {
            $headerKey = $request->hasHeader($apiKeyParam) ? $apiKeyParam : self::API_KEY_HEADER_DEFAULT;
            $publicKey = $request->header($headerKey);
        }

        if (empty($publicKey) && in_array('route', $authTypes)) {
            $publicKey = $request->get($apiKeyParam);
        }

        if (empty($publicKey) && in_array('session', $authTypes)) {
            $publicKey = app('session')->get($apiKeyParam);
        }

        if (empty($publicKey)) {
            throw new UnauthorizedHttpException('No auth token provided', 'No auth token provided');
        }

        $apiProvider = config('api_core.api_keys_class');
        if (!$apiProvider) {
            throw new UnauthorizedHttpException("Please, configure 'api_core.api_keys_class'", 'No provider configured');
        }

        $apiKey = $apiProvider::findByKey($publicKey);

        if (empty($apiKey) || !$apiKey instanceof ApiKeyInterface) {
            throw new UnauthorizedHttpException('Wrong public key', 'Wrong public key');
        }

        $route = $request->route();
        if (is_array($route)) {
            $route = &$this->getArrayRouteRef(app());
            $route[2] += ['apiKey' => $apiKey];
        } else {
            $route->setParameter('apiKey', $apiKey);
        }

        if (method_exists($apiKey, 'authenticate')) {
            $apiKey->authenticate();
        }

        return tap($next($request), function ($response) use ($request, $apiKey) {
            if (method_exists($apiKey, 'registerResponse')) {
                $apiKey->registerResponse($request, $response);
            }
        });
    }

    /**
     * Gets a reference to an array representing the route in App, instead of a copy
     * 
     * @param mixed $app
     * @return array
     */
    protected function &getArrayRouteRef($app)
    {
        return Closure::bind(function & () {
            return $this->currentRoute;
        }, $app, $app)->__invoke();
    }

}
