<?php

namespace InSegment\ApiCore\Models;

use InSegment\ApiCore\Services\AttributeCacheStore;

class AttributeCache
{
    const CACHE_HAS_CAST = 1 << 1;
    const CACHE_HAS_METHOD = 1 << 2;
    const CACHE_BASE_HAS_METHOD = 1 << 3;
    const CACHE_HAS_GET_MUTATOR = 1 << 4;
    const CACHE_HAS_SET_MUTATOR = 1 << 5;
    const CACHE_IS_DATE_ATTRIBUTE = 1 << 6;
    const CACHE_IS_JSON_CASTABLE = 1 << 7;
    const CACHE_FILL_JSON = 1 << 8;
    const CACHE_IN_DATES = 1 << 9;
    
    /**
     * Date format cache
     *
     * @var array [string $connectionName => mixed $dateFormat]
     */
    public static $dateFormatCache = [];
    
    /**
     * Runtime cache contents
     * 
     * @var array [string $attribute => bool $cachedValue]
     */
    public $hasCast = [];
    public $hasMethod = [];
    public $baseHasMethod = [];
    public $hasGetMutator = [];
    public $hasSetMutator = [];
    public $isDateAttribute = [];
    public $isJsonCastable = [];
    public $fillJson = [];
    public $inDates = [];
    
    /**
     * Is the store freshly created/loaded
     *
     * @var bool 
     */
    public $fresh = true;

    /**
     * Cast type cached
     *
     * @var array [string $attribute => mixed $castType]
     */
    public $castType = [];

    /**
     * Casts cached
     *
     * @var array|null
     */
    public $casts;
    
    /**
     * Default table name cached there
     *
     * @var string|null
     */
    public $table;
    
    /**
     * Resolved connection
     * 
     * @var \Illuminate\Database\Connection
     */
    public $connection;
    
    /**
     * Resolved connection name
     *
     * @var string
     */
    public $connectionName;
    
    /**
     * Address of the store in cache
     *
     * @var string
     */
    protected $class;
    protected $boxName;
    
    /**
     * Constructor
     * 
     * @param string $class
     * @param string $boxName
     */
    public function __construct(string $class, string $boxName)
    {
        $this->class = $class;
        $this->boxName = $boxName;
    }
    
    /**
     * Load one attribute from mask
     * 
     * @param string $attribute
     * @param int $mask
     * @return null
     */
    public function unpack(string $attribute, int $mask)
    {
        $this->hasCast[$attribute] = $mask & self::CACHE_HAS_CAST;
        $this->hasMethod[$attribute] = $mask & self::CACHE_HAS_METHOD;
        $this->baseHasMethod[$attribute] = $mask & self::CACHE_BASE_HAS_METHOD;
        $this->hasGetMutator[$attribute] = $mask & self::CACHE_HAS_GET_MUTATOR;
        $this->hasSetMutator[$attribute] = $mask & self::CACHE_HAS_SET_MUTATOR;
        $this->isDateAttribute[$attribute] = $mask & self::CACHE_IS_DATE_ATTRIBUTE;
        $this->isJsonCastable[$attribute] = $mask & self::CACHE_IS_JSON_CASTABLE;
        $this->fillJson[$attribute] = $mask & self::CACHE_FILL_JSON;
        $this->inDates[$attribute] = $mask & self::CACHE_IN_DATES;
    }
    
    /**
     * Cache characteristics for an attribute
     * 
     * @param string $attribute
     * @return null
     */
    public function cacheAttribute(string $attribute)
    {
        AttributeCacheStore::getInstance()->updateMask($this->class, $this->boxName, $attribute,
            ($this->hasCast[$attribute] ? self::CACHE_HAS_CAST : 0) |
            ($this->hasMethod[$attribute] ? self::CACHE_HAS_METHOD : 0) |
            ($this->baseHasMethod[$attribute] ? self::CACHE_BASE_HAS_METHOD : 0) |
            ($this->hasGetMutator[$attribute] ? self::CACHE_HAS_GET_MUTATOR : 0) |
            ($this->hasSetMutator[$attribute] ? self::CACHE_HAS_SET_MUTATOR : 0) |
            ($this->isDateAttribute[$attribute] ? self::CACHE_IS_DATE_ATTRIBUTE : 0) |
            ($this->isJsonCastable[$attribute] ? self::CACHE_IS_JSON_CASTABLE : 0) |
            ($this->fillJson[$attribute] ? self::CACHE_FILL_JSON : 0) |
            ($this->inDates[$attribute] ? self::CACHE_IN_DATES : 0)
        );
    }
    
}
