<?php

namespace InSegment\ApiCore\Models;

use Illuminate\Support\Facades\DB;

use InSegment\ApiCore\Interfaces\AutoKeyRemapInterface;
use InSegment\ApiCore\Traits\SliceSchemaTrait;

class GenerationsMatch extends SupplementaryTable
{
    use SliceSchemaTrait;
  
    /**
     * Whether it is known that the table exists
     *
     * @var bool
     */
    protected static $tableExists = false;
    
    /**
     * Last-known count the table have
     *
     * @var int|null
     */
    protected static $iteration = 0;
    
    /**
     * Constructor
     * 
     * @param array $attributes
     * @param string|null $uid
     */
    public function __construct(array $attributes = array())
    {
        parent::__construct($attributes);
        
        $this->table = "{$this->getSchema()}.{$this->getTable()}";
    }
    
    /**
     * Run the migrations.
     *
     * @param array $options [
     *     'uid' => string|null,
     *     'inMemory' => bool|null,
     *     'temporary' => bool|null
     * ]
     * @return null
     */
    public static function establish(array $options)
    {
        if (!static::$tableExists) {
            static::createSupplementary($options, [
                "`uuid` BIGINT(20) UNSIGNED NOT NULL",
                "`autokey_value` INT(10) UNSIGNED NOT NULL",
                "`iteration` INT(10) UNSIGNED NOT NULL",
                "KEY `generations_match_iteration_index` (`iteration`)"
            ]);
            
            static::$tableExists = true;
        }
    }
    
    /**
     * Register new autoKey identifiers into remap table
     * 
     * @param \InSegment\ApiCore\Interfaces\AutoKeyRemapInterface $remapModel
     * @param string $realTable
     * @param string $tempTable
     * @param string $autoKey
     * @return int
     */
    public function match(AutoKeyRemapInterface $remapModel, string $realTable, string $tempTable, string $autoKey)
    {
        static::establish(['temporary' => true]);
        
        $myTable = $this->getTable();
        list($database, $tableName) = explode('.', $myTable);
        
        DB::statement("SET @ai = (LAST_INSERT_ID() - 1)");
        
        $uuidField = $remapModel->uuidFieldName();
        $autoKeyField = $remapModel->autoKeyFieldName();
        $generationsTbl = $remapModel->getTable();
        $iteration = ++static::$iteration;
        
        $generationsMatchQuery = DB
            ::table("{$tempTable} as temp")
            ->select([
                "temp.{$autoKey} as uuid",
                DB::raw("(@ai := (@ai + 1)) as `autokey_value`"),
                DB::raw("{$iteration} as `iteration`")
            ])
            ->orderBy("temp.{$autoKey}", 'asc');
            
        $matchedAmount = DB::affectingStatement(
            "INSERT INTO `{$database}`.`{$tableName}` (`uuid`, `autokey_value`, `iteration`) {$generationsMatchQuery->toSql()}",
            $generationsMatchQuery->getBindings()
        );
            
        $updatedMatchesAmount = static
            ::where("{$myTable}.iteration", '=', $iteration)
            ->join("{$generationsTbl} as gens", "gens.{$uuidField}", "=", "{$myTable}.uuid")
            ->update(["gens.{$autoKeyField}" => DB::raw("`{$database}`.`{$tableName}`.`autokey_value`")]);
            
        if ($matchedAmount !== $updatedMatchesAmount) {
            throw new \Exception("When matching generations to auto-increment keys, matched amount did not become equal to updated amount!");
        }
            
        return $updatedMatchesAmount;
    }
    
}
