<?php

namespace InSegment\ApiCore\Models;

use Illuminate\Support\Facades\DB;

use InSegment\ApiCore\Models\SliceRemap;
use InSegment\ApiCore\Models\SupplementarySliceTable;
use InSegment\ApiCore\Exceptions\BaseSliceException;

use InSegment\ApiCore\Interfaces\SliceManagementInterface;
use InSegment\ApiCore\Traits\SliceManagementTrait;
use InSegment\ApiCore\Traits\SliceSchemaTrait;

use InSegment\ApiCore\Services\SliceMerger\MergeManager;
use InSegment\ApiCore\Services\ParseCreateTable;

class SliceOperation extends BaseOperation implements SliceManagementInterface
{
    use SliceManagementTrait;
    use SliceSchemaTrait;
    
    /**
     * Create new slice from UID
     * 
     * @param string|null $UID
     * @param bool $create
     * @return \InSegment\ApiCore\Models\SliceOperation
     * @throws \InSegment\ApiCore\Exceptions\BaseSliceException
     */
    public static function fromUID($UID, bool $create = true)
    {
        if ($create) {
            $result = static::create(['uid' => $UID]);
        } else {
            $result = new static(['uid' => $UID]);
        }
        
        SupplementarySliceTable::$sliceManager = $result;
        return $result;
    }

    /**
     * Create new slice with a new UID
     * 
     * @param bool $create
     * @return \InSegment\ApiCore\Models\SliceOperation
     * @throws \InSegment\ApiCore\Exceptions\BaseSliceException
     */
    public static function newSlice(bool $create = true)
    {
        $sliceUID = DB::selectOne('SELECT UUID_SHORT() as `uid`')->uid;
        if (!$sliceUID) {
            throw (new BaseSliceException(BaseSliceException::CODE_UNABLE_TO_GENERATE_UID))->compile();
        }
        
        return static::fromUID($sliceUID, $create);
    }
    
    /**
     * Fillable attributes
     * 
     * @var array
     */
    protected $fillable = [
        'uid'
    ];
    
    /**
     * SliceMerger instance
     * 
     * @var \InSegment\ApiCore\Services\SliceMerger\MergeManager
     */
    protected $sliceMerger;
    
    /**
     * Instance of show create table parser
     * 
     * @var \InSegment\ApiCore\Services\ParseCreateTable
     */
    protected $showCreateParser;
    
    /**
     * Counters for written and new records in the Slice (for estimates)
     *
     * @var array [
     *     'new' => [
     *         string $table => int,
     *         ...
     *     ]
     *     'written' => [
     *         string $table => int,
     *         ...
     *     ]
     * ] 
     */
    protected $counters = [];
    
    /**
     * Counters for written and new records in the Slice (for estimates)
     *
     * @param array $counters [
     *     'new' => [
     *         string $table => int,
     *         ...
     *     ]
     *     'written' => [
     *         string $table => int,
     *         ...
     *     ]
     * ] 
     */
    public function setCounters(array $counters)
    {
        $this->counters = $counters;
        
        return $this;
    }
    
    /**
     * Increase a counter for a table
     * 
     * @param string $type 'written' or 'new'
     * @param string $table
     * @param int $amount
     * @return $this
     */
    public function increaseTableCounter(string $type, string $table, int $amount)
    {
        if (isset($this->counters[$type][$table])) {
            $this->counters[$type][$table] += $amount;
        } else {
            $this->counters[$type][$table] = $amount;
        }
        
        if ($this->counters[$type][$table] < 0) {
            throw new \Exception('Counters must be greater or equal to 0');
        }
        
        return $this;
    }
    
    /**
     * Get table for remaps
     * 
     * @return string
     */
    public function getRemapsTable()
    {
        return (new SliceRemap([], $this->uid))->getTable();
    }
    
    /**
     * Get ParseCreateTable instance
     * 
     * @return \InSegment\ApiCore\Services\ParseCreateTable
     */
    public function getShowCreateParser()
    {
        return $this->showCreateParser ?? ($this->showCreateParser = ParseCreateTable::getInstance());
    }

    /**
     * Get SliceMerger instance
     * 
     * @param array $types
     * @return \InSegment\ApiCore\Services\SliceMerger\MergeManager
     */
    public function getMergeManager($types = [MergeManager::TEMP_TABLE_TYPE_INSERT])
    {
        return $this->sliceMerger ?? ($this->sliceMerger = new MergeManager($this, $this->getShowCreateParser(), $types));
    }

    /**
     * Get slice operation UID
     * 
     * @return string
     */
    public function getUID(): string
    {
        return $this->uid;
    }
    
    /**
     * Get count of records was written for table
     * 
     * @param string $table
     * @return int
     */
    public function getWrittenCount(string $table): int
    {
        return $this->counters['written'][$table];
    }
    
    /**
     * Get count of records was inserted into table
     * 
     * @param string $table
     * @return int
     */
    public function getNewCount(string $table): int
    {
        return $this->counters['new'][$table];
    }

    /**
     * Clear created temporary tables for the given UUID and tables names
     *
     * @param string|int $uuid
     * @param array $tables
     * @return bool
     */
    public static function clearUUID($uuid, array $tables): bool
    {
        $sliceOperation = static::find($uuid);
        if ($sliceOperation === null) {
            return true;
        }

        if ($sliceOperation->status === static::STATUS_DONE) {
            return false;
        }

        $mergeManager = $sliceOperation->getMergeManager(MergeManager::DEFAULT_TEMP_TABLE_TYPES);

        foreach ($mergeManager->listTemporaryTableNames($tables) as $tableName => $tempTableDef) {
            $mergeManager->dropTempTable($tempTableDef);
        }

        $sliceOperation->delete();
        return true;
    }
}
