<?php

namespace InSegment\ApiCore\Models;

use Illuminate\Support\Facades\DB;

use InSegment\ApiCore\Services\MysqlCommands;
use InSegment\ApiCore\Traits\SliceSchemaTrait;
use InSegment\ApiCore\Models\SliceOperation;
use InSegment\ApiCore\Models\UUIDGeneration;

class SliceStructure
{

    use SliceSchemaTrait;
    
    /**
     * @var string
     */
    protected $uid;
    
    /**
     * Tables to temp tables mapping
     *
     * @var array [
     *      string $tableName => [
     *          string $type => string $tempTableOfType
     *      ],
     *      ...
     * ],
     */
    protected $tables;
    
    /**
     * SQL code used for creation of temp tables
     *
     * @var string[]|null
     */
    protected $sql;
    
    /**
     * Constructor
     * 
     * @param array $structure ['uid' => string, 'tables' => array, 'sql' => string[]]
     * @throws \Exception
     */
    public function __construct(array $structure)
    {
        $keys = array_keys($structure);
        
        if (($missing = array_diff(['uid', 'tables', 'sql'], $keys))) {
            $implodeMissing = implode(', ', $missing);
            throw new \Exception("Missing necessary fields in response: {$implodeMissing}");
        }
        
        $this->uid = $structure['uid'];
        $this->tables = $structure['tables'];
        $this->sql = $structure['sql'];
    }
    
    
    /**
     * Get uid of requested structure
     * 
     * @return string|null
     */
    public function getUid()
    {
        return $this->uid;
    }
    
    /**
     * Get tables of requested structure
     * 
     * @return array [
     *      string $tableName => [
     *          string $type => string $tempTableOfType
     *      ],
     *      ...
     * ],
     */
    public function getTables()
    {
        return $this->tables;
    }
    
    /**
     * Get tables of requested structure prefixed by current transaction schema
     * 
     * @return array [
     *      string $tableName => [
     *          string $type => string $tempTableOfType
     *      ],
     *      ...
     * ],
     */
    public function getTablesInSchema()
    {
        $tempTables = [];
        
        foreach ($this->tables as $targetTable => $unprefixedTemps) {
            foreach ($unprefixedTemps as $type => $tempName) {
                $tempTables[$targetTable][$type] = "{$this->getSchema()}.{$tempName}";
            }
        }
        
        return $tempTables;
    }
    
    /**
     * Get all necessary temporary tables
     * 
     * @return string[]
     */
    public function getAllTemps()
    {
        $allTempTables = [];
        
        list($generationDb, $generationTable) = explode('.', (new UUIDGeneration([], $this->uid))->getTable());
        $allTempTables[] = $generationTable;
        
        foreach ($this->getTables() as $tempTables) {
            foreach ($tempTables as $tempTable) {
                $allTempTables[] = $tempTable;
            }
        }
        
        return $allTempTables;
    }
    
    /**
     * Get SQL code from requested structure
     * 
     * @return string[]|null
     */
    public function getSql()
    {
        return $this->sql;
    }
    
    /**
     * Execute previously requested SQL
     * 
     * @throws \Exception
     */
    public function execSql()
    {
        if (!$this->sql) {
            throw new \Exception("Structure must have sql before executing!");
        }
        
        $insertOperation = new SliceOperation(['uid' => $this->uid]);
        
        if (!$insertOperation->save()) {
            throw new \Exception("Cannot save slice operation with uid: {$this->uid}!");
        }
        
        foreach ($this->sql as $sql) {
            DB::statement($sql);
        }
    }
    
    /**
     * Check that necessary tables exists
     * 
     * @throws \Exception
     */
    public function checkTables()
    {
        $schema = $this->getSchema();
        $allTemps = $this->getAllTemps();

        $tablesExists = DB
            ::table('information_schema.tables')
            ->select('table_name')
            ->where('table_schema', '=', $schema)
            ->whereIn('table_name', $allTemps)
            ->pluck('table_name')
            ->toArray();
        
        if (($missingTables = array_diff($allTemps, $tablesExists))) {
            $implodeMissing = implode(', ', $missingTables);
            throw new \Exception("Expected these tables, but they are missing: {$implodeMissing}");
        }
    }
    
    /**
     * Export tables
     * 
     * @param \InSegment\ApiCore\Services\MysqlCommands $commands
     * @param resource|null $outputResuoutceHandler
     * @return string|int|null
     */
    public function exportTables(MysqlCommands $commands, $outputResuoutceHandler = null)
    {
        return $commands->mysqldump(
            $this->getSchema(),
            $this->getAllTemps(),
            $outputResuoutceHandler
        );
    }
    
    /**
     * Import tables
     * CAUTION: Never use this method with user input!
     * 
     * @param \InSegment\ApiCore\Services\MysqlCommands $commands
     * @param resource|string|\Symfony\Component\Process\InputStream $input
     * @param callable|null $callback
     * @throws \Exception
     */
    public function importTables(MysqlCommands $commands, $input, $callback = null)
    {
        $commands->mysql($this->getSchema(), $input, $callback);
        $this->checkTables();
    }
    
}
