<?php

namespace InSegment\ApiCore\Models;

use InSegment\ApiCore\Interfaces\SADConsumer;

/**
 * A model of non-recursive iteration of models, their relations and attributes
 */
class SourceAttributeDownstream
{
    const ACCEPT_STATE_INITIAL = 0;
    const ACCEPT_STATE_ACCEPTED = 1;
    const ACCEPT_STATE_SAVED = 2;
    const ACCEPT_STATE_SOURCE_START = 3;
    const ACCEPT_STATE_READY_TO_SAVE = 4;
    
    /**
     * Source. It is an array of Models which need to be processes
     * 
     * @var \InSegment\ApiCore\Models\SADSource
     */
    protected $currentSource;
    
    /**
     * Array of stacked sources
     *
     * @var \InSegment\ApiCore\Models\SADSource[]
     */
    protected $sourceStack = [];
    
    /**
     * Iterating state
     *
     * @var bool 
     */
    protected $isIterating = false;
    
    /**
     * The main cycle. Accepts SADConsumer, array of Models, and a class for attribute rules
     * What the consumer considers 'models', 'attributes' or 'downstream' is irrelevant, also by default
     * the main cycle won't add any new sources or downstream in process, so the consumer should call
     * methods SourceAttributeDownstream::newSource and SourceAttributeDownstream::addDownStream to affect the cycle
     * 
     * Each times the consumer method is called, we stop operating inside that loop and wait for next one.
     * This is done to ensure, that if the consumer method added the new source and the loop is changed, we will
     * react to that immediately.
     * 
     * NOTE: the reason around this "Mighty Mega-function"
     * 
     * ALL the recursion of model relations is done THERE. The array of Models get into Source, they are processed by
     * the Consumer, the Consumer gives new Models and the loop switches to process them on the next iteration. Also
     * they add downstream attributes. Everything else we know beforehand.
     * 
     * The cycle has a switch inside which changes its position in the statements depending on that position
     * and which both affects and is affected by the loop in which it operates. So these entities are tight bound together.
     * 
     * The only two variables the method has in its scope is $this and $consumer, which do not change, but are
     * required on EVERY step of the cycle. So there is no sense to create anything else in visibility.
     * 
     * So we can abstract the manipulation of arrays in properties of $this into methods of $this. But these methods
     * should be protected anyway, because the outside intervention into the arrays in properties of $this is dangerous.
     * And there is no real reason to extend the class, because anything done in another way is not going to work there.
     * So of source we can increase effectiveness of the method calls by declaring the class final,
     * but still the method resolution will be as slow as 7 times incrementing a variable.
     * 
     * Having no extra calls to just manipulate arrays or retrieve values there, we eliminate that overhead completely.
     * And again, the loop in theory works with every model in the collection, with every relation of that model,
     * with every attribute of that model, sometimes twice per attribute, and with every model on the every
     * collection of the every model of the collection, and so on.
     * 
     * If the function grows past 300 lines of code, consider writing another model of non-recursive iteration
     * for the cause which required to write that much.
     * 
     * END NOTE.
     * 
     * @param \InSegment\ApiCore\Interfaces\SADConsumer $consumer
     * @return null
     */
    public function iterate(SADConsumer $consumer)
    {
        if ($this->isIterating) {
            throw new \Exception('SourceAttributeDownstream iteration intervention detected. Iteration should not be called recursively inside another iteration');
        }
        
        if (!$this->currentSource) {
            return;
        }
        
        $this->isIterating = true;
        
        // as the consumer is allowed to modify state of our cycle, and each of supplied sources must be still processed,
        // method SourceAttributeDownstream::newSource saves the previous state into the stack
        // the outer cycle handles restoration of the state saved into stack once the current source has no more entries
        do {
            // this construction allows to avoid recursion by modification of the state of the cycle
            // the while cycle there iterates through the source, which is possibly changed on almost each iteration
            while ($this->currentSource->intIndex < $this->currentSource->total) {
                // note that on each step consumer's method can and is intended to change OUR cycle's state
                // so we cannot just hop to the next case, after each call of consumer's method we must jump to
                // the while and recheck the state
                switch ($this->currentSource->state) {
                    
                    // before consumer is informed of model
                    case self::ACCEPT_STATE_INITIAL:
                        if ($this->currentSource->intIndex === 0) {
                            $this->currentSource->state = self::ACCEPT_STATE_SOURCE_START;
                            $consumer->sourceStarted($this->currentSource->params);
                        } else {
                            $this->currentSource->state = self::ACCEPT_STATE_ACCEPTED;
                            $consumer->acceptModel(
                                $this->currentSource->arrayItems[$this->currentSource->intIndex],
                                $this->currentSource->params
                            );
                        }
                    break;
                    
                    // start of processing source
                    case self::ACCEPT_STATE_SOURCE_START:
                        $this->currentSource->state = self::ACCEPT_STATE_ACCEPTED;
                        $consumer->acceptModel(
                            $this->currentSource->arrayItems[$this->currentSource->intIndex],
                            $this->currentSource->params
                        );
                    break;
                
                    // processing of attributes 
                    case self::ACCEPT_STATE_ACCEPTED:
                        if ($this->currentSource->attrIndex < $this->currentSource->attribute->total) {
                            // pass attribute to the consumer and advance attribute iteration simultaneously
                            $consumer->acceptAttribute(
                                $this->currentSource->attribute->arrayItems[
                                    // micro-optimization: increment index in the same spot where it is used,
                                    // so we don't have to store the result in $attribute variable, and can pass it
                                    // directly to the $consumer::acceptAttribute
                                    $this->currentSource->attrIndex++
                                ],
                                $this->currentSource->params
                            );
                        } else {
                            $this->currentSource->state = self::ACCEPT_STATE_READY_TO_SAVE;
                        }
                    break;
                                    
                    // then save of the model
                    case self::ACCEPT_STATE_READY_TO_SAVE:
                        $this->currentSource->state = self::ACCEPT_STATE_SAVED;
                        $consumer->saveModel(
                            $this->currentSource->arrayItems[$this->currentSource->intIndex],
                            $this->currentSource->params
                        );
                    break;
                
                    // the model is saved
                    case self::ACCEPT_STATE_SAVED:
                        // advance to the next model and check if the source ends
                        if (++$this->currentSource->intIndex === $this->currentSource->total) {
                            // if this call results in value strictly equal to false, we do not contunue the loop
                            // in that case, break out of the most outer loop (from switch, from while, from do...while)
                            if ($consumer->sourceEnded($this->currentSource->params) === false) {
                                break 3;
                            }
                        } else {
                            $this->currentSource->state = self::ACCEPT_STATE_INITIAL;
                            $this->currentSource->attrIndex = 0; // reset attributes iteration
                        }
                    break;
                }
            }
        } while (
            // if some source is stacked, we restore the stacked source, otherwise the loop will exit
            $this->currentSource = array_pop($this->sourceStack)
        );
        
        $this->isIterating = false;
    }
    
    /**
     * Add new array of Models so it will become the current source and stack everything down
     * 
     * @param \InSegment\ApiCore\Models\SADSource $source
     * @return null
     */
    public function newSource(SADSource $source)
    {
        $this->sourceStack[] = $this->currentSource;
        $this->currentSource = $source;
    }
    
    /**
     * Add something to the end of the downstream
     * 
     * @param mixed $value
     */
    public function addDownStream($value)
    {
        // TODO: remove
    }
}
