<?php

namespace InSegment\ApiCore\Models;

use InSegment\ApiCore\Traits\Instantiatible;
use InSegment\ApiCore\Services\BufferedBuilder;
use InSegment\ApiCore\Traits\SliceSchemaTrait;

abstract class SupplementarySliceTable extends SupplementaryTable
{
    use Instantiatible;
    use SliceSchemaTrait;
    
    /**
     * Slice manager of the slice operation
     *
     * @var \InSegment\ApiCore\Interfaces\SliceManagementInterface
     */
    public static $sliceManager;
    
    /**
     * Original table name
     *
     * @var string
     */
    protected $defaultTable;
    
    /**
     * UID
     *
     * @var string
     */
    public $uid;
    
    /**
     * Boot method
     * Subscribe to instantiation event
     * 
     * @return null
     */
    public static function boot()
    {
        parent::boot();
        
        static::instantiated(function ($model, $instantiator) {
            if (isset($instantiator['model']) && $instantiator['model']->uid !== $model->uid) {
                $model->uid = $instantiator['model']->uid;
                $model->updateTable();
            }
        });
    }
    
    /**
     * Constructor
     * 
     * @param array $attributes
     * @param string|null $uid
     */
    public function __construct(array $attributes = array(), $uid = null)
    {
        parent::__construct($attributes);
        $this->defaultTable = $this->getTable();
        $this->uid = $uid ?? self::$sliceManager->getUID();
        $this->updateTable();
    }
    
    /**
     * Update table of the model
     * 
     * @return $this
     */
    protected function updateTable()
    {
        $database = $this->getSchema();
        $table = $this->tableNameGenerator();
        $this->table = "{$database}.{$table}";
        return $this;
    }

    /**
     * Table name generator
     * 
     * @return string
     */
    protected function tableNameGenerator()
    {
        return substr(implode('_', [$this->defaultTable, $this->uid]), 0, 64);
    }
    
    /**
     * Create a new Eloquent query builder for the model.
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @return \InSegment\ApiCore\Services\BufferedBuilder|static
     */
    public function newEloquentBuilder($query)
    {
        return new BufferedBuilder($query);
    }
    
    /**
     * Clear the push query
     * 
     * @return null
     */
    public static function clearPushQuery()
    {
        unset(static::$pushQueries[static::class]);
    }

}
