<?php

namespace InSegment\ApiCore\Models;

use InSegment\ApiCore\Traits\EventsReplace;
use InSegment\ApiCore\Traits\CachesAttributes;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Eloquent\Model;

/**
 * With generic model the database is holding structure and data and the model's task is to represent it (table).
 * With supplementary table - the table in database exists only to support some computations and the model
 * decides what is the structure in the database and what is the data, and it is a model's task to make sure
 * the structure and data in DB is up to date.
 * Example of usage:
 * - multisession temporary table
 * - table with the given amount of rows (aka table of rows)
 * - when you need multiple tables of some structure with different generated names, based on business logic
 */
abstract class SupplementaryTable extends Model
{
    use EventsReplace;
    use CachesAttributes;
    
    /**
     * Indicate there is no incrementing key
     * 
     * @var bool 
     */
    public $incrementing = false;

    /**
     * The model has timestamps
     * 
     * @var bool
     */
    public $timestamps = false;
    
    /**
     * @var array
     */
    protected static $defaultOptions = [];
    
    /**
     * Run the migrations.
     *
     * @param array $options
     * @return null
     */
    public abstract static function establish(array $options);

    /**
     * Reverse the migrations.
     *
     * @param string|null $uid
     * @return null
     */
    public static function disband($uid = null)
    {
        Schema::dropIfExists((new static([], $uid))->getTable());
    }
    
    /**
     * Get query optionally specifying $uid
     * 
     * @param string|null $uid
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public static function query($uid = null)
    {
        return (new static([], $uid))->newQuery();
    }
 
    /**
     * Check that the supplementary table exists
     * 
     * @param string|null $uid
     * @return bool
     */
    protected static function existsSupplementary($uid = null): bool
    {
        list($schema, $table) = explode('.', (new static([], $uid))->getTable());
        $selection = DB::select("select * from `information_schema`.`tables` where `table_schema` = ? and `table_name` = ?", [$schema, $table]);
        return count($selection) > 0;
    }
    
    /**
     * Execute create table statement with specified options and fields
     * 
     * @param array $options [
     *     'uid' => string|null,
     *     'inMemory' => bool|null,
     *     'temporary' => bool|null,
     *     'returnSql' => bool|null,
     * ]
     * @param array $fields
     * @return string|null
     */
    protected static function createSupplementary(array $options, array $fields)
    {
        $options += static::$defaultOptions;
        list($schema, $table) = explode('.', (new static([], $options['uid'] ?? null))->getTable());
        if (!empty($options['postfix'])) {
            $table .= $options['postfix'];
        }
        
        $typeOfTable = ($options['temporary'] ?? false) ? 'TEMPORARY TABLE' : 'TABLE';
        $engine = ($options['inMemory'] ?? false) ? 'MEMORY' : 'InnoDB';
        
        $implodeStatements = implode(",\n  ", $fields);
        
        $sql = <<<SQL
CREATE {$typeOfTable} `{$schema}`.`{$table}` (
  {$implodeStatements}
) ENGINE={$engine} DEFAULT CHARSET=utf8
SQL;
        if ($options['returnSql'] ?? false) {
            return $sql;
        }
        
        DB::statement($sql);
    }
    
    /**
     * @param array $options
     */
    public static function setDefaultOptions(array $options)
    {
        static::$defaultOptions = $options;
    }
    
    
}
