<?php

namespace InSegment\ApiCore\Models;

use Illuminate\Support\Facades\DB;

class Transaction extends BaseOperation
{
    /**
     * Fillable attributes
     * 
     * @var array
     */
    protected $fillable = [
        'uid',
        'status',
        'session_id',
        'root_class',
        'clean_status'
    ];
    
    /**
     * Boot method
     * 
     * @return null
     */
    public static function boot()
    {
        parent::boot();
        
        static::saved(function (Transaction $transaction) {
            if (($dataModel = $transaction->getDataModel())) {
                $dataModel->setUID($transaction->uid);
            }
            
            if (!$transaction->status) {
                $transaction->status = static::STATUS_ACTIVE;
            }
        });
    }
    
    /**
     * Transaction data
     *
     * @var \InSegment\ApiCore\Models\TransactionData
     */
    protected $dataModel;
    
    /**
     * Get TransactionData
     * 
     * @return \InSegment\ApiCore\Models\TransactionData
     */
    public function getDataModel()
    {
        return $this->dataModel;
    }
    
    /**
     * Set TransactionData
     * 
     * @return $this
     */
    public function setDataModel(TransactionData $dataModel)
    {
        $this->dataModel = $dataModel;
        return $this;
    }
    
    /**
     * Is Transaction active
     * 
     * @return bool
     */
    public function isActive()
    {
        return $this->getAttribute('status') === static::STATUS_ACTIVE;
    }
    
    /**
     * Is Transaction not prepared
     * 
     * @return bool
     */
    public function isNotPrepared()
    {
        return !$this->getAttribute('status');
    }
    
    /**
     * Prepare first step of Transaction
     * 
     * @param string $sessionId
     * @param string $rootClass
     * @return $this
     */
    public function prepareTransaction($sessionId, $rootClass)
    {
        $this->status = static::STATUS_STALE;
        $this->uid = DB::selectOne('SELECT UUID_SHORT() as `uid`')->uid;
        $this->session_id = $sessionId;
        $this->root_class = $rootClass;
        $this->save();
        return $this;
    }
    
    /**
     * Activate Transaction (from this moment Transactor would start to treat requests with data import as valid)
     * 
     * @return $this
     */
    public function activateTransaction()
    {
        $this->status = static::STATUS_ACTIVE;
        $this->save();
        return $this;
    }
    
    /**
     * Finish the Transaction
     * 
     * @param bool $clear
     * @return $this
     */
    public function finishTransaction($clear)
    {
        if ($this->exists && $this->status === static::STATUS_ACTIVE) {
            $this->status = static::STATUS_DONE;
            $this->save();
        }
        
        if (($dataModel = $this->getDataModel())) {
            $clear && $dataModel->clearData();
            $dataModel->clearSession();
        }
        
        return $this;
    }
    
    /**
     * Abort the Transaction
     * 
     * @param bool $clear
     * @return $this
     */
    public function abortTransaction($clear)
    {
        if ($this->exists && $this->status === static::STATUS_ACTIVE) {
            $this->status = static::STATUS_ABORT;
            $this->save();
        }
        
        if ($clear && ($dataModel = $this->getDataModel())) {
            $dataModel->clearData();
            $dataModel->clearSession();
        }
        
        return $this;
    }
}
