<?php

namespace InSegment\ApiCore\Models;

use Carbon\Carbon;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Exception;

class TransactionData
{
    const TABLE_NAME = 'transaction_counts';
    
    const DATA_TEMPLATE = [
        'received' => [],
        'written' => [],
        'new' => [],
        'dyn_uw' => [],
        'reserved' => []
    ];
    
    /**
     * Transaction UID (if present)
     *
     * @var string
     */
    protected $uid;
    
    /**
     * Session instance
     *
     * @var \Illuminate\Contracts\Session\Session
     */
    protected $session;
    
    /**
     * Transaction
     *
     * @var \InSegment\ApiCore\Models\Transaction 
     */
    protected $transaction;
    
    /**
     * Session-set objects
     *
     * @var array
     */
    protected $sessionObjects;
    
    /**
     * Internal representation of the TransactionData
     *
     * @var array 
     */
    protected $data = self::DATA_TEMPLATE;
    
    /**
     * The representation currently in the DB
     *
     * @var array 
     */
    protected $inDbState = self::DATA_TEMPLATE;
    
    /**
     * The state is not loaded from DB
     *
     * @var bool 
     */
    protected $notLoaded = true;
    
    /**
     * Date format for Carbon
     *
     * @var mixed 
     */
    protected $dateFormat;
    
    /**
     * Constructor
     * Automatically loads session, and Transaction if possible and it is active
     */
    public function __construct()
    {
        $this->dateFormat = DB::query()->grammar->getDateFormat();
        $this->session = app('session');
        $objects = $this->session->get('transaction-objects');
        $this->sessionObjects = is_array($objects) ? $objects : [];
        
        $UID = $this->session->get('transaction-uid');
        
        if ($UID && ($transaction = Transaction::find($UID))) {
            $this->uid = $UID;
            $this->transaction = $transaction;
            if ($transaction->status === Transaction::STATUS_ACTIVE) {
                $transaction->touch();
            }
        } else {
            $this->transaction = new Transaction;
        }
        
        $this->transaction->setDataModel($this);
    }
    
    /**
     * Whether the transaction UID is set
     * 
     * @return bool
     */
    public function hasUID(): bool
    {
        return $this->uid !== null;
    }
    
    /**
     * Get Transaction UID
     * 
     * @return mixed
     */
    public function getUID()
    {
        return $this->uid;
    }
    
    /**
     * Get data
     * 
     * @param string $key
     * @return array
     */
    public function getData(string $key = null)
    {
        return isset($key) ? $this->data[$key] : $this->data;
    }
    
    /**
     * Check the table in dynamic underweights
     * 
     * @param string $table
     * @return bool
     */
    public function getIsDynamicUnderweight(string $table)
    {
        return $this->data['dyn_uw'][$table] ?? false;
    }
    
    /**
     * Get reserved amount for table
     * 
     * @param string $table
     * @return int
     */
    public function getReservedAmount(string $table)
    {
        return $this->data['reserved'][$table] ?? 0;
    }
    
    /**
     * Get received amount for table
     * 
     * @param string $table
     * @return int
     */
    public function getRecievedAmount(string $table)
    {
        return $this->data['received'][$table] ?? 0;
    }
    
    /**
     * Get written amount for table
     * 
     * @param string $table
     * @return int
     */
    public function getWrittenAmount(string $table)
    {
        return $this->data['written'][$table] ?? 0;
    }
    
    /**
     * Get new amount for table
     * 
     * @param string $table
     * @return int
     */
    public function getNewAmount(string $table)
    {
        return $this->data['new'][$table] ?? 0;
    }
    
    /**
     * Get session objects
     * 
     * @return array
     */
    public function getSessionObjects()
    {
        return $this->sessionObjects;
    }
    
    /**
     * Get Transaction
     * 
     * @return \InSegment\ApiCore\Models\Transaction
     */
    public function getTransaction()
    {
        return $this->transaction;
    }
    
    /**
     * Get session object of the specified class
     * 
     * @param string $class
     * @return mixed
     */
    public function getSessionObject(string $class) {
        return Arr::get($this->sessionObjects, $class);
    }
    
    /**
     * Set UID
     * 
     * @param mixed $uid
     */
    public function setUID($uid)
    {
        if ($this->transaction->uid !== $uid) {
            throw new Exception("UID in TransactionData must be exactly same as in Transaction!");
        }
        
        $this->uid = $uid;
        $this->session->put(['transaction-uid' => $uid]);
    }
    
    /**
     * Set dynamic underweight data
     * 
     * @param array $data
     */
    public function setDynamicUnderweightData(array $data)
    {
        $this->data['dyn_uw'] = $data;
    }
    
    /**
     * Get session object of the specified class
     * 
     * @param string $class
     * @param mixed $object
     * @return null
     */
    public function setSessionObject(string $class, $object = null) {
        if ($object == null) {
            unset($this->sessionObjects[$class]);
        } else {
            $this->sessionObjects[$class] = $object;
        }
        
        $this->session->put(['transaction-objects' => $this->sessionObjects]);
    }
    
    /**
     * Load TransactionData from DB
     * 
     * @return $this
     * @throws Exception
     */
    public function loadData()
    {
        if ($this->uid !== null) {
            $dataFromTable = DB::table(self::TABLE_NAME)->where('uid', $this->uid)->get();
            foreach ($dataFromTable as $rowFromTable) {
                $table = $rowFromTable->table;
                $amount = $rowFromTable->amount;
                switch($rowFromTable->count_type) {
                    case 'received': $this->data['received'][$table] = (int) $amount; break;
                    case 'written': $this->data['written'][$table] = (int) $amount; break;
                    case 'new': $this->data['new'][$table] = (int) $amount; break;
                    case 'dyn_uw': $this->data['dyn_uw'][$table] = (bool) $amount; break;
                    case 'reserved': $this->data['reserved'][$table] = (int) $amount; break;
                    default: throw new Exception("Unknown 'count_type' from table data of TransactionData: '{$rowFromTable->count_type}'");
                }
            }
            
            $this->inDbState = $this->data;
        }
        
        return $this;
    }
    
    /**
     * Save TransactionData to DB
     * 
     * @return $this
     */
    public function saveData()
    {
        if ($this->uid !== null && ($insertStatement = $this->getStoreInsertStatement())) {
            DB::transaction(function () use ($insertStatement) {
                $table = self::TABLE_NAME;
                DB::statement("SELECT 1 FROM `{$table}` WHERE `uid` = ? FOR UPDATE", [$this->uid]);

                if ($this->checkIsOutOfSync()) {
                    throw new Exception("Transaction data is out of sync. Cannot continue");
                }

                DB::affectingStatement(...$insertStatement);
                $this->inDbState = $this->data;
                $this->deleteObsoleteData();
            });
        }
        
        return $this;
    }
    
    /**
     * Clear session data
     * 
     * @return $this
     */
    public function clearSession()
    {
        $this->session->put(['transaction-uid' => null]);
        $this->session->put(['transaction-objects' => null]);
    }
    
    /**
     * Clear TransactionData in DB
     * 
     * @return $this
     */
    public function clearData()
    {
        if ($this->uid !== null && !$this->notLoaded) {
            $table = self::TABLE_NAME;
            DB::statement("DELETE FROM `{$table}` WHERE `uid` = ?", [$this->uid]);
            $this->inDbState = self::DATA_TEMPLATE;
        }
        
        return $this;
    }
    
    /**
     * Set reserved amount for table
     * 
     * @param string $table
     * @param int|null $amount
     * @return $this
     * @throws Exception
     */
    public function setReservedAmount(string $table, int $amount = null)
    {
        if (!empty($this->data['received'][$table]) || !empty($this->data['written'][$table]) || !empty($this->data['new'][$table])) {
            throw new Exception("Cannot reset 'reserved' amount for table '{$table}', because its counts are already started");
        }
        
        if (isset($amount)) {
            $this->data['reserved'][$table] = $amount;
        }
        
        $this->data['received'][$table] = 0;
        $this->data['written'][$table] = 0;
        $this->data['new'][$table] = 0;
        return $this;
    }
    
    /**
     * If the received amount is ready to increment
     * 
     * @param string $table
     * @return bool
     */
    public function canReceiveIncrementing(string $table)
    {
        return isset($this->data['reserved'][$table]) && $this->data['received'][$table] < $this->data['reserved'][$table];
    }
    
    /**
     * Increment amounts for table
     * 
     * @param string $table
     * @param bool $isWritten
     * @param bool $isNew
     * @param bool $isIncrementing
     * @return $this
     * @throws Exception
     */
    public function incrementReceived(string $table, bool $isWritten, bool $isNew, bool $isIncrementing)
    {
        if ($isIncrementing && !isset($this->data['reserved'][$table])) {
            throw new Exception("Table '{$table}' has no reserved amount to be incremented!");
        }
        
        ++$this->data['received'][$table];
        if ($isWritten) {
            ++$this->data['written'][$table];
        }
        
        if ($isNew) {
            ++$this->data['new'][$table];
        }
        
        return $this;
    }
    
    /**
     * Get statement for DB to insert with update of data
     * 
     * @return [$sql, $bindings]|false
     */
    protected function getStoreInsertStatement()
    {
        $insertColumns = '`uid`, `count_type`, `table`, `amount`, `created_at`, `updated_at`';
        $now = Carbon::now()->format($this->dateFormat);
        $bindings = [];
        $inserts = [];
        
        foreach ($this->data as $countType => $tablesToCounts) {
            foreach (array_diff_assoc($tablesToCounts, $this->inDbState[$countType]) as $countTable => $amount) {
                $inserts[] = '(?, ?, ?, ?, ?, ?)';
                $bindings[] = $this->uid; // `uid`
                $bindings[] = $countType; // `count_type`
                $bindings[] = $countTable; // `table`
                $bindings[] = $amount; // `amount`
                $bindings[] = $now; // `created_at`
                $bindings[] = $now; // `updated_at`
            }
        }
        
        if (!count($inserts)) {
            return false;
        }
        
        $table = self::TABLE_NAME;
        $odkuColumns = '`amount` = VALUES(`amount`), `updated_at` = VALUES(`updated_at`)';
        $implodeInserts = implode(', ', $inserts);
        $sql = "INSERT INTO `{$table}` ({$insertColumns}) VALUES {$implodeInserts} ON DUPLICATE KEY UPDATE {$odkuColumns}";
        return [$sql, $bindings];
    }
    
    /**
     * Check the DB that the data was not modified by another process
     * 
     * @return bool
     */
    protected function checkIsOutOfSync()
    {
        $table = self::TABLE_NAME;
        list($checksCondition, $bindings) = $this->getConditionsForStoredData();
        return (bool) DB::select("SELECT EXISTS (SELECT 1 FROM `{$table}` WHERE `uid` = ? AND {$checksCondition}) as `exists`", $bindings)[0]->exists;
    }
    
    /**
     * Check the DB that the data was not modified by another process
     * 
     * @return int
     */
    protected function deleteObsoleteData()
    {
        $table = self::TABLE_NAME;
        list($checksCondition, $bindings) = $this->getConditionsForStoredData();
        return DB::affectingStatement("DELETE FROM `{$table}` WHERE `uid` = ? AND {$checksCondition}", $bindings);
    }
    
    /**
     * Get check conditions and bindings for data we have stored
     * 
     * @return [$checksCondition, $bindings]
     */
    protected function getConditionsForStoredData()
    {
        $checks = [];
        $bindings = [$this->uid];
        foreach ($this->inDbState as $countType => $tablesToCounts) {
            foreach ($tablesToCounts as $countTable => $amount) {
                $checks[] = '?, ?, ?';
                $bindings[] = $countType;
                $bindings[] = $countTable;
                $bindings[] = $amount;
            }
        }
        
        $implodeChecks = implode('), (', $checks);
        $checksCondition = $checks ? "(`count_type`, `table`, `amount`) NOT IN (({$implodeChecks}))" : '0 = 1';
        return [$checksCondition, $bindings];
    }
}
