<?php

namespace InSegment\ApiCore\Models;

class TraverseState
{
    /**
     * Data
     *
     * @var mixed
     */
    public $data;
    
    /**
     * Class of the Model being imported
     *
     * @var string|null
     */
    public $class;
    
    /**
     * Model being imported/exported
     * 
     * @var \Illuminate\Database\Eloquent\Model|null
     */
    public $model;
    
    /**
     * Currently processed mapping
     * 
     * @var array|null
     */
    public $mapping;
    
    /**
     * If the parent Model have relation to current Model, it is then used
     * to affect current Model attributes by constraints
     * 
     * @var \Illuminate\Database\Eloquent\Relations\Relation|null
     */
    public $relation;
    
    /**
     * Relation name to the Model being imported from the model up the stack
     *
     * @var string|null
     */
    public $relationName;
    
    /**
     * Import/export mode
     *
     * @var int 
     */
    public $mode;
    
    /**
     * Level of stacking
     *
     * @var int
     */
    protected $level = 0;
    
    /**
     * Stack
     *
     * @var array 
     */
    protected $stack = [];
    
    /**
     * If the collection is initialised on the level, it contains data from which to iterate it
     *
     * @var array [
     *     int $level => [
     *         'data' => & array,
     *         'keys' => array,
     *         'total' => int,
     *         'max' => int|string,
     *         'i' => int
     *     ]
     * ] 
     */
    protected $collections = [];
    
    /**
     * Constructor
     * 
     * @param mixed $data
     */
    public function __construct($data)
    {
        $this->data = $data;
    }
    
    /**
     * Repoint data without making the level deeper
     * 
     * @param array $dataAddress
     * @return null
     */
    public function repointData($dataAddress)
    {
        foreach ($dataAddress as $partOfAdderss) {
            $this->data = & $this->data[$partOfAdderss];
        }
    }
    
    /**
     * Stack the snapshot of the current level, then make it deeper
     * 
     * @return null
     */
    public function stackNewLevel()
    {
        $this->stack[] = $this->getMoment();
        ++$this->level;
    }
    
    /**
     * Un-stack the current level: decrement the current depth and restore the snapshot from the stack
     * If the level is greater than 1, uninitialised collection
     * 
     * @throws \Exception
     * @return null
     */
    public function unstackLevel()
    {
        if (!$this->level) {
            throw new \Exception("Unexpected unstacking: already at the ground level");
        }
        
        if ($this->level > 1) {
            unset($this->collections[$this->level]);
        }
        
        --$this->level;
        $this->embraceMoment(array_pop($this->stack));
    }
    
    /**
     * Initialise collection on the current data level
     * If the collection is already initialised, do not replace it
     * 
     * @param array $keys
     * @throws \Exception
     * @return null
     */
    public function initCollection(array $keys = null)
    {
        if (!is_array($this->data)) {
            if (isset($this->data)) {
                throw new \Exception("Expected data to be array, got something else");
            } else {
                $this->data = [];
            }
        }
        
        if (!isset($this->collections[$this->level])) {
            $total = count($this->data);
            
            if (!isset($keys)) {
                $keys = array_keys($this->data);
                $max = $total ? max($keys) : 0;
            } else {
                $max = max($keys);
            }
            
            $this->collections[$this->level] = [
                'data' => & $this->data,
                'keys' => $keys,
                'total' => $total,
                'max' => $max,
                'i' => 0
            ];
        } else if (isset($keys) && !$this->collections[$this->level]['total']) {
            $this->collections[$this->level]['keys'] = $keys;
            $this->collections[$this->level]['max'] = max($keys);
        }
        
        unset($this->data);
    }
    
    /**
     * Clear the result on the current data level
     * 
     * @return null
     */
    public function clearResult()
    {
        if (isset($this->data)) {
            if (isset($this->collections[$this->level])) {
                if ($this->data === []) {
                    return;
                }

                $this->data = [];
                $this->collections[$this->level] = [
                    'data' => & $this->data,
                    'keys' => [],
                    'total' => 0,
                    'max' => 0,
                    'i' => 0
                ];
            } else {
                $this->data = null;
            }
        }
    }
    
    /**
     * Advance pointer of collection iteration
     * If the collection is fully iterated, add new key to collection
     * 
     * @param array|null $keyValues
     * @return null
     */
    public function advanceCollection(array $keyValues = null)
    {
        if (isset($this->collections[$this->level])) {
            $i = $this->collections[$this->level]['i'];
            if (isset($keyValues)) {
                if ($i >= $this->collections[$this->level]['total']) {
                    ++$this->collections[$this->level]['total'];
                }
                
                $this->data = & $this->collections[$this->level]['data'];
                foreach ($keyValues as $keyValue) {
                    if (!isset($this->data)) {
                        $this->data = [];
                    }
                    
                    if (!isset($this->data[$keyValue])) {
                        $this->data[$keyValue] = null;
                    }
                    
                    $this->data = & $this->data[$keyValue];
                }
            } else {
                if ($i < $this->collections[$this->level]['total']) {
                    $key = $this->collections[$this->level]['keys'][$i];
                } else {
                    // extend the array of collection and increase the max key
                    // if it happens to be a string key, it will make 'a' to 'b', but 'z' to 'aa' and so on like columns in excel
                    ++$this->collections[$this->level]['total'];
                    if (isset($this->collections[$this->level]['keys'][$i])) {
                        $key = $this->collections[$this->level]['keys'][$i];
                    } else {
                        $this->collections[$this->level]['keys'][$i] = $key = $this->collections[$this->level]['max']++;
                    }

                    $this->collections[$this->level]['data'][$key] = null;
                }
                
                $this->data = & $this->collections[$this->level]['data'][$key];
            }
            
            ++$this->collections[$this->level]['i'];
        }
    }
    
    /**
     * Collect snapshot of the traverse state
     * 
     * @return array
     */
    public function getMoment()
    {
        return [
            'data' => & $this->data,
            'relationName' => $this->relationName,
            'relation' => $this->relation,
            'model' => $this->model,
            'class' => $this->class,
            'mapping' => $this->mapping,
            'mode' => $this->mode
        ];
    }
    
    /**
     * Restore snapshot of traverse state
     * 
     * @param array $moment
     * @return null
     */
    public function embraceMoment(array $moment)
    {
        $this->data = & $moment['data'];
        $this->relationName = $moment['relationName'];
        $this->relation = $moment['relation'];
        $this->model = $moment['model'];
        $this->class = $moment['class'];
        $this->mapping = $moment['mapping'];
        $this->mode = $moment['mode'];
    }
    
}
