<?php

namespace InSegment\ApiCore\Models;

use Illuminate\Support\Facades\DB;
use InSegment\ApiCore\Interfaces\AutoKeyRemapInterface;

class UUIDGeneration extends SupplementarySliceTable implements AutoKeyRemapInterface
{
    protected $primaryKey = 'uuid';
    protected $autoKeyField = 'id';
    
    /**
     * Run the migrations.
     *
     * @param array $options [
     *     'tables' => string[],
     *     'uid' => string|null,
     *     'inMemory' => bool|null,
     *     'temporary' => bool|null
     * ]
     * @return string|null
     */
    public static function establish(array $options)
    {
        $tablesEnum = "'" . implode("', '", $options['tables']) . "'";
        
        return static::createSupplementary($options, [
            "`id` INT(10) UNSIGNED DEFAULT NULL",
            "`uuid` BIGINT(20) UNSIGNED NOT NULL",
            "`table` ENUM ({$tablesEnum}) NOT NULL",
            "`used` TINYINT(1) UNSIGNED NOT NULL DEFAULT 0",
            "PRIMARY KEY `uuid_generation_primary` (`uuid`)",
            "KEY `uuid_generation_table_index` (`table`)",
            "KEY `uuid_generation_used_index` (`used`)",
        ]);
    }

    /**
     * Name of the auto-key field in remap
     * 
     * @return string
     */
    public function autoKeyFieldName(): string
    {
        return $this->autoKeyField;
    }

    /**
     * Name of the UUID field in remap
     * 
     * @return string
     */
    public function uuidFieldName(): string
    {
        return $this->primaryKey;
    }
    
    /**
     * Get statement generating amount if ids for table and key
     * 
     * @param string $table
     * @param int $rows
     * @param string|null $uid
     * @return null
     */
    public static function generateForTable(string $table, $rows, $uid = null)
    {
        $nRowsQuery = TableOfRows
            ::take($rows)
            ->select([])
            ->selectRaw('UUID_SHORT() as `uuid`')
            ->selectRaw("? as `table`", [$table])
            ->toBase();
        
        list($schema, $myTable) = explode('.', (new static([], $uid))->getTable());
        DB::statement("INSERT INTO `{$schema}`.`{$myTable}` (`uuid`, `table`) {$nRowsQuery->toSql()}", $nRowsQuery->getBindings());
    }

    /**
     * Dispose rows which was not used to remap something
     * 
     * @return null
     */
    public function disposeUnused()
    {
        list($schema, $myTable) = explode('.', $this->getTable());
        DB::statement("DELETE FROM `{$schema}`.`{$myTable}` WHERE `used` = 0");
    }
    
}
