<?php

namespace InSegment\ApiCore\Monads;

use Closure;

final class Maybe
{
    /**
     * A currently held value
     * 
     * @var mixed 
     */
    private $value;
    
    /**
     * A context which will be bound to Closures $this
     *
     * @var mixed 
     */
    private $context;
    
    /**
     * A class used to bind context to Closures
     * 
     * @var string
     */
    private $contextClass;
    
    /**
     * Constructor
     * 
     * @param mixed $value
     * @param mixed $context
     */
    private function __construct($value = null, $context = null)
    {
        $this->value = $value;
        $this->context = $context;
        $this->contextClass = $context ? get_class($context) : null;
    }
    
    /**
     * Create a Maybe for specified value and context. If the value is Maybe, it is returned itself
     * 
     * @param \App\Monads\Maybe $value
     * @param mixed $context
     * @return \App\Monads\Maybe
     */
    public static function of($value = null, $context = null): Maybe
    {
        if ($value instanceof Maybe) {
            return $value;
        }
        
        return new Maybe($value, $context);
    }
    
    /**
     * Set a new value
     * 
     * @param \App\Monads\Maybe $value
     * @param bool $strict
     * @return $this
     */
    public function set($value, bool $strict = false)
    {
        if (!$strict && $value instanceof Maybe) {
            $this->value = $value->value;
        } else {
            $this->value = $value;
        }
        
        return $this;
    }
    
    /**
     * Change a value (if present) to the result of applying Closure to the currently held value
     * If there is no value, Closure would not be called
     * 
     * @param \Closure $function
     * @return \App\Monads\Maybe
     */
    public function map(Closure $function): Maybe
    {
        if (isset($this->value)) {
            $value = $this->call($function, true);
            
            if ($value instanceof Maybe) {
                $value->context = $this->context;
                return $value;
            } else {
                $this->value = $value;
            }
        }
        
        return $this;
    }
    
    /**
     * Return a new Maybe with or without currently held value based on filter Closure
     * 
     * @param \Closure $predicate
     * @return \App\Monads\Maybe
     */
    public function filter($predicate): Maybe
    {
        return new Maybe($this->call($predicate, true) ? $this->value : null, $this->context);
    }
    
    /**
     * Execute a Closure if there is some value held with this value as an argument
     * 
     * @param \Closure $consumer
     * @return \App\Monads\Maybe
     */
    public function with(Closure $consumer): Maybe
    {
        isset($this->value) && $this->call($consumer, true);

        return $this;
    }
    
    /**
     * Execute a Closure if there is is some value held
     * 
     * @param \Closure $runnable
     * @return \App\Monads\Maybe
     */
    public function having(Closure $runnable): Maybe
    {
        isset($this->value) && $this->call($runnable);

        return $this;
    }
    
    /**
     * Execute a Closure if there is no value held
     * 
     * @param \Closure $runnable
     * @return \App\Monads\Maybe
     */
    public function missing(Closure $runnable): Maybe
    {
        !isset($this->value) && $this->call($runnable);
        
        return $this;
    }
    
    /**
     * Return the held value or an alternative if no such present
     * if $strict it will retuen just the specified argument, otherwise if the argument is \Closure,
     * it will return result of its execution, or if it is \Throwable, it will throw it
     * 
     * @param \Closure|\Throwable\mixed $else
     * @param bool $strict
     * @return mixed
     * @throws \Throwable
     */
    public function orElse($else, bool $strict = false)
    {
        if (isset($this->value)) {
            return $this->value;
        }
        
        if ($strict) {
            return $else;
        }
        
        $ret = $else instanceof Closure ? $else() : $else;
        
        if ($ret instanceof \Throwable) {
            throw $ret;
        }
        
        return $ret;
    }
    
    /**
     * If there is no held value, set one specified
     * if $strict it will retuen just the specified argument, otherwise if the argument is \Closure,
     * it will set value to result of its execution
     * 
     * @param \Closure|mixed $value
     * @param bool $strict
     * @return \App\Monads\Maybe
     */
    public function orMaybe($value = null, bool $strict = false): Maybe
    {
        if (isset($this->value)) {
            return $this;
        }
        
        if ($strict) {
            return $this->set($value);
        } else {
            return $this->set($value instanceof Closure ? $this->call($value) : $value);
        }
    }
    
    /**
     * Combine current and the next Maybe with a combiner Closure which will receive
     * values of both Maybe's and which return will be wrapped into a Maybe
     * 
     * @param \App\Monads\Maybe $nextValue
     * @param \Closure $combiner
     * @return \App\Monads\Maybe
     */
    public function andThen($nextValue, Closure $combiner)
    {
        if ($nextValue instanceof Maybe) {
            $nextMaybe = $nextValue;
        } else {
            $nextMaybe = Maybe::of($nextValue, $this->context);
        }
        
        if (isset($this->context)) {
            $combiner = $combiner->bindTo($this->context, $this->contextClass);
        }
        
        return Maybe::of(call_user_func($combiner, $this, $nextMaybe), $this->context);
    }
    
    /**
     * Call a Closure binding context to it 
     * 
     * @param \Closure $function
     * @param bool $value
     * @return mixed
     */
    private function call($function, bool $value = false)
    {
        if (isset($this->context)) {
            $function = $function->bindTo($this->context, $this->contextClass);
        }
        
        if (!$value) {
            return call_user_func($function);
        }
        
        return call_user_func($function, $this->value);
    }
}
