<?php

namespace InSegment\ApiCore\Scopes;

use Illuminate\Support\Facades\DB;

use InSegment\ApiCore\Services\Transactor;

use Illuminate\Database\Query\Builder as QueryBuiler;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Scope;

/**
 * A global scope to ease the use of Transactor with Eloquent by the help of UNIONs
 */
class TransactionSearchScope implements Scope
{
    /**
     * Properties of the query Bilder which will be shared between UNIONS
     * 
     * @var type 
     */
    private static $builderPropertiesToAssociate = [
        'connection', 'grammar', 'processor', 'bindings', 'aggregate', 'wheres'
    ];

    /**
     * A set of classes having this scope added
     * 
     * @var array
     */
    private static $applied = [];
    
    /**
     * The indication of scope is enabled
     * 
     * @var array
     */
    private static $enabled = [];
    
    /**
     * Adds the global scope to the class, if it wasn't already added there
     * 
     * @return null
     */
    public static function applyTo(string $class)
    {
        if (!Transactor::doesHaveTables($class)) {
            return;
        }

        if (!self::isApplied($class)) {
            $class::addGlobalScope(new static($class));
            self::$applied[$class] = true;
        }

        self::$enabled[$class] = true;
    }
    
    /**
     * Checks if the scope is applied to the class
     * 
     * @param string $class
     * @return bool
     */
    public static function isApplied(string $class): bool
    {
        return isset(self::$applied[$class]);
    }
    
    /**
     * Checks if the scope is applied to the class and enabled
     * 
     * @param string $class
     * @return bool
     */
    public static function isEnabled(string $class): bool
    {
        return self::isApplied($class) && self::$enabled[$class];
    }

    /**
     * Disable the scope
     */
    public static function disable()
    {
        foreach (array_keys(self::$applied) as $class) {
            self::$enabled[$class] = false;
        }
    }
    
    /**
     * Class of the Eloquent Model this scope works on
     * 
     * @var string 
     */
    private $applyClass;
    
    /**
     * Private to prevent creation from outside
     * Sets applyClass
     * 
     * @param string $class
     */
    private function __construct(string $class) {
        $this->applyClass = $class;
    }
    
    /**
     * Adds UNION on the three tables: the main table, the update table and the insert table
     * No modifications possible while under this global scope.
     * I.e, you will need $query->withoutGlobalScope(TransactionSearchScope::class) to be able to UPDATE or INSERT
     * 
     * UPDATE table takes priority over the main table and INSERT table.
     * INSERT table takes priority over the main table
     * Main table is the last resort
     * 
     * @param \Illuminate\Database\Eloquent\Builder $builder
     * @param \Illuminate\Database\Eloquent\Model $model
     * @return null
     */
    public function apply(Builder $builder, Model $model)
    {
        if (!self::isEnabled(get_class($model))) {
            return;
        }

        $table = $model->getTable();

        // Produce a hack subquery in from clause
        // Instead of selecting directly from the table, select from union of three transactional tables
        // Produce an anonymous class for from clause instead of a string, so it will be called (stringifyed) LATE,
        // after all other scopes are already applied and the query Builder is actually ready to retrieve from DB
        $builder
            ->select("{$table}.*")
            ->from(DB::raw(new class($this, $builder, $table) {
                private $scope;
                private $builder;
                private $table;

                public function __construct(TransactionSearchScope $scope, Builder $builder, $table)
                {
                    $this->scope = $scope;
                    $this->builder = $builder;
                    $this->table = $table;
                }

                public function __toString()
                {
                    $lateQueryWithUnions = $this->scope->makeQueryWithUnions($this->builder, $this->table);
                    return "({$lateQueryWithUnions->toSql()}) as `{$this->table}`";
                }
            }))
            ->groupBy("{$table}.{$model->getKeyName()}");
    }
    
    /**
     * Select from sumquery with unions
     * 
     * @param \Illuminate\Database\Eloquent\Builder $builder
     * @param string $table
     * @return \Illuminate\Database\Query\Builder
     */
    public function makeQueryWithUnions(Builder $builder, $table)
    {
        $baseQuery = $builder->getQuery();

        $updateTable = Transactor::getUpdateTable($this->applyClass);
        $updateQuery = $this->modifyQuery(DB::table("{$updateTable} as {$table}"), $baseQuery);

        $insertTable = Transactor::getInsertTable($this->applyClass);
        $insertQuery = $this->modifyQuery(DB::table("{$insertTable} as {$table}"), $baseQuery);

        $mainQuery = $this->modifyQuery(DB::table($table), $baseQuery);

        return $updateQuery
            ->unionAll($insertQuery)
            ->unionAll($mainQuery);
    }
    
    /**
     * Hack several builders into sharing conditions and bindings
     * 
     * @param \Illuminate\Database\Query\Builder $target
     * @param \Illuminate\Database\Query\Builder $source
     * @return \Illuminate\Database\Query\Builder
     */
    private function modifyQuery(QueryBuiler $target, QueryBuiler $source)
    {
        foreach (self::$builderPropertiesToAssociate as $property) {
            $target->$property = &$source->$property;
        }
        
        return $target;
    }
    
}