<?php

namespace InSegment\ApiCore\Services\Api2Api;

use Illuminate\Support\Str;
use InSegment\ApiCore\Exceptions\ApiRequestException;
use InSegment\ApiCore\Models\SliceStructure;
use InSegment\ApiCore\Services\Logger;

class Connection
{

    /**
     * @var \InSegment\ApiCore\Services\Api2Api\CommunicationClientInterface 
     */
    protected $client;
    
    /**
     * @var \InSegment\ApiCore\Services\Logger
     */
    protected $logger;

    /**
     * @var string
     */
    protected $apiKey;
    
    /**
     * @var string
     */
    protected $apiUrl;
    
    /**
     * @var bool
     */
    protected $secure;

    /**
     * Default is False.
     * @var bool
     */
    protected $authorized = false;

    /**
     * Default is False.
     * @var bool
     */
    protected $transactionAcquired = false;

    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Services\Api2Api\ConnectorInterface $connector
     * @param \InSegment\ApiCore\Services\Api2Api\EndpointInterface $endpoint
     * @param \InSegment\ApiCore\Services\Logger|null $logger
     */
    public function __construct(ConnectorInterface $connector, EndpointInterface $endpoint, Logger $logger = null)
    {
        if ($logger !== null) {
            $this->setLogger($logger);
        }

        $this->apiKey = $endpoint->getApiKey();
        $this->apiUrl = $endpoint->getApiURL();
        $this->secure = Str::startsWith($this->apiUrl, 'https://') || config('api_core.api_2_api_allow_unsafe_http') === true;
        
        $connector->setUseCookies();
        $connector->setDefaultHeaders([
            'Accept' => 'application/json',
            // Is used for urls like "title/by-keyword", etc.
            'api-key' => $this->apiKey,
        ]);
        
        $this->client = $connector->getClient();
    }
    
    /**
     * 
     * @param InSegment\ApiCore\Services\Logger $logger
     */
    public function setLogger(Logger $logger)
    {
        $this->logger = $logger;
    }
    
    /**
     * Whether the connection is made over https
     * 
     * @return bool
     */
    public function isSecure()
    {
        return $this->secure;
    }

    /**
     * POSTs request to CM API and returns a response.
     * 
     * @param string $url "export/company", "title/by-keyword", etc.
     * @param array $data Will be sent in JSON format.
     * @param bool $needAuth Optional. Default - True
     * @return array ['code' => 123, 'message' => '', 'data' => 'raw data']
     */
    protected function post(string $url, array $data, bool $needAuth = true): array
    {
        if ($needAuth && !$this->authorized) {
            $logResponseOnly = true;
            $this->__request('POST', 'auth', ['json' => ['api_key' => $this->apiKey]], $logResponseOnly);
            $this->authorized = true;
        }

        return $this->__request('POST', $url, ['json' => $data]);
    }

    protected function get(string $url, $data): array
    {
        return $this->__request('GET', $url, ['query' => $data]);
    }

    /**
     * Perform API request through connection client
     * 
     * @param string $type
     * @param string $url
     * @param array $data
     * @param bool $logResponseOnly [Optional] False by default.
     * @return array ['code' => 123, 'message' => '', 'data' => 'raw data'], or empty array if $data is empty.
     */
    protected function request(string $type, string $url, array $data, bool $logResponseOnly = false): array
    {
        return $this->__request($type, $url, ['json' => $data], $logResponseOnly);
    }

    protected function __request(string $method, string $uri, array $options, bool $logResponseOnly = false): array
    {
        $log = !is_null($this->logger);
        if ($log) {
            $logTag = str_replace('/', '-', $uri);
            if (!$logResponseOnly) {
                $this->logger->push("{$logTag} > data", $options);
            }
        }

        try {
            $response = $this->client->request($method, "{$this->apiUrl}{$uri}", $options);
            $responseBodyContents = $response->getBody()->getContents();
            $responseDecoded = json_decode($responseBodyContents, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                if ($log) {
                    $this->logger->push("{$logTag} > response", $responseBodyContents);
                    $this->logger->close();
                }

                throw (new ApiRequestException(ApiRequestException::CODE_WRONG_RESPONSE))->compile();
            }
        } catch (\Throwable $e) {
            $log && $this->client->logExceptionDetails($this->logger, $e, $logTag);
            throw $e;
        }

        $log && $this->logger->push("{$logTag} > response", $responseDecoded);

        return [
            'code' => $response->getStatusCode(),
            'message' => $response->getReasonPhrase(),
            'data' => $responseDecoded,
        ];
    }
    
    /**
     * Request remote slice structure giving ability to execute its statements
     * 
     * @param array $param
     * @return \InSegment\ApiCore\Models\SliceStructure|bool
     * @throws \Exception
     */
    public function requestStructure(array $param)
    {
        // Avoid man-in-the middle attacks to be able to inject any SQL code they want
        // 
        // TODO: also refactor key-as-password based authentication to public/private key verifiation handshake
        // as an admin key-password compromentation would make this code still vulnerable
        if (!$this->isSecure()) {
            throw new \Exception("Requesting SQL code for execution is only allowed over https");
        }
        
        $param['api_key'] = $this->apiKey;
        $response = $this->post('get-slice-structure', $param, false);
        
        if ($this->isBadResponseWithUndo($response['data'])) {
            return false;
        }
        
        return new SliceStructure($response['data']['data']);
    }    

    /**
     *
     * @param array $count
     * @param string $root
     * @return bool
     */
    public function transactionAcquire(array $count, string $root = null): bool
    {
        $data = ['count' => $count];
        if ($root !== null) {
            $data['root'] = $root;
        }

        $response = $this->post('transaction/acquire', $data);

        $this->transactionAcquired = true;

        if ($this->isBadResponseWithUndo($response['data'])) {
            $this->transactionAcquired = false;
            return false;
        }

        return true;
    }

    /**
     * 
     * @param bool $returnResult Optional. Wether or not to return the result of transaction. Default - False.
     * @return mixed
     */
    public function transactionClose(bool $returnResult = false)
    {
        $response = $this->post('transaction/close', []);

        if ($this->isBadResponseWithUndo($response['data'])) {
            return false;
        }

        if ($returnResult) {
            return $response['data'];
        }

        return true;
    }

    /**
     * 
     * @return bool
     */
    public function transactionUndo()
    {
        $response = $this->post('transaction/undo', []);

        return !$this->isBadResponse($response['data']);
    }

    /**
     * 
     * @param array $data
     * @return bool
     */
    protected function isBadResponseWithUndo($data): bool
    {
        if (!$this->isBadResponse($data)) {
            return false;
        }

        if ($this->transactionAcquired) {
            $this->transactionUndo();
        }

        if (!is_null($this->logger)) {
            $this->logger->close();
        }

        return true;
    }
    
    /**
     * 
     * @param array $data
     * @return bool
     */
    protected function isBadResponse($data): bool
    {
        return !(isset($data['status']) && $data['status'] == 200);
    }
    
}
