<?php

namespace InSegment\ApiCore\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\App;

use InSegment\ApiCore\Models\AttributeCache;
use InSegment\ApiCore\Services\MicroEvent;

class AttributeCacheStore
{

    /**
     * An instance
     * 
     * @var \InSegment\ApiCore\Services\AttributeCacheStore 
     */
    private static $instance = null;
    
    /**
     * Runtime cache stores for multiple classes
     *
     * @var array
     */
    public $boxes;
    
    /**
     * Prefix for cache
     *
     * @var string 
     */
    protected $prefix;
    
    /**
     * Packed caches for multiple classes
     *
     * @var array
     */
    protected $cache;
    
    /**
     * Is the caching of attributes enabled
     * 
     * @var bool
     */
    protected $enabled;
    
    /**
     * Whether the handler to write cache is set on the termination event
     *
     * @var bool
     */
    protected $isHandlerSet = false;
    
    /**
     * Whether the application is down for maintenance
     *
     * @var bool
     */
    protected $isDownForMaintenance;

    /**
     * A singleton, no instantiation from outside
     */
    private function __construct()
    {
        $this->prefix = $this->getCachePrefix();
        $this->enabled = $this->getIsEnabled();
    }
    
    /**
     * Get attribute cache store if it exists
     * 
     * @param string $class
     * @param string $boxName
     * @return \InSegment\ApiCore\Models\AttributeCache
     */
    public function newStore(string $class, string $boxName)
    {
        if (!isset($this->boxes)) {
            $this->boot();
        }
        
        return $this->boxes[$class][$boxName] ?? ($this->boxes[$class][$boxName] = new AttributeCache($class, $boxName));
    }
    
    /**
     * Update mask of an attribute and set handler to cache it if necessary
     * 
     * @param string $class
     * @param string $boxName
     * @param string $attribute
     * @param int $newMask
     * @return null
     */
    public function updateMask(string $class, string $boxName, string $attribute, int $newMask)
    {
        if ($newMask !== ($this->cache[$class][$boxName][$attribute] ?? null)) {
            $this->cache[$class][$boxName][$attribute] = $newMask;
            
            if (!$this->isHandlerSet) {
                MicroEvent::registerEvent('InSegment\ApiCore\Middleware\TerminationEvent', 'termination', function () {
                    if ($this->enabled) {
                        Cache::forever($this->prefix, $this->cache);
                    }
                });

                $this->isHandlerSet = true;
            }
        }
    }
    
    /**
     * Try to boot cache from the caching storage and use cache for attribute of models of $class
     * 
     * @param string $class
     * @param string $box
     * @param string $attribute
     * @return null
     */
    protected function boot()
    {
        $this->boxes = [];
        
        if ($this->enabled) {
            $this->cache = Cache::get($this->prefix) ?? [];
            
            foreach ($this->cache as $cachedClass => $cachedBoxes) {
                foreach ($cachedBoxes as $cachedBox => $packedFlagsByAttribute) {
                    $this->boxes[$cachedClass][$cachedBox] = $newStore = new AttributeCache($cachedClass, $cachedBox);
                    foreach ($packedFlagsByAttribute as $attribute => $mask) {
                        $newStore->unpack($attribute, $mask);
                    }
                }
            }
        } else {
            $this->cache = [];
        }
    }
    
    /**
     * Get cache prefix
     * 
     * @return string
     */
    protected function getCachePrefix()
    {
        return config('api_core.cache_prefix', 'insegment.api-core.') . 'attribute-cache';
    }
    
    /**
     * Get whether the caching is enabled
     * 
     * @return string
     */
    protected function getIsEnabled()
    {
        return config('api_core.attibutes_cache') && !App::isDownForMaintenance();
    }
    
    /**
     * Get AttributeCacheStore instance
     * 
     * @return \InSegment\ApiCore\Services\AttributeCacheStore
     */
    public static function getInstance()
    {
        if (self::$instance == null) {
            self::$instance = new AttributeCacheStore();
        }
        
        return self::$instance;
    }
    
}
