<?php

namespace InSegment\ApiCore\Services;

use Illuminate\Database\Query\Expression;
use Illuminate\Database\Eloquent\Builder;

use InSegment\ApiCore\Services\BufferedWriter;

/**
 * Buffers records being written to database if the buffer is set up
 * 
 * Buffer can be BufferedWriter instance, or some object with method to get it (i.e. Transactor)
 */
class BufferedBuilder extends Builder
{
    /**
     * Transactor deciding whether the Model can be modified by this BufferedBuilder
     * 
     * @var \InSegment\ApiCore\Services\BufferedWriter
     */
    public static $buffer;
    
    /**
     * Insert a new record into the database.
     *
     * @param  array  $values
     * @return bool
     */
    public function insert(array $values)
    {
        if (self::$buffer) {
            $model = $this->getModel();
            self::$buffer->put(BufferedWriter::STATEMENT_INSERT, get_class($model), $model->getTable(), $values);
            return true;
        } else {
            return parent::insert($values);
        }
    }

    /**
     * Insert a new record and get the value of the primary key.
     *
     * @param  array   $values
     * @param  string|null  $sequence
     * @return int
     */
    public function insertGetId(array $values, $sequence = null)
    {
        if (self::$buffer) {
            $model = $this->getModel();
            self::$buffer->put(BufferedWriter::STATEMENT_INSERT, get_class($model), $model->getTable(), $values);
            return $model->getKey();
        } else {
            return parent::insertGetId($values, $sequence);
        }
    }

    /**
     * Remove all of the expressions from a list of bindings.
     *
     * @param  array  $bindings
     * @return array
     */
    public function cleanBindings(array $bindings)
    {
        return array_values(array_filter($bindings, function ($binding) {
            return ! $binding instanceof Expression;
        }));
    }

    /**
     * Update a record in the database.
     *
     * @param  array  $values
     * @return int
     */
    public function update(array $values)
    {
        if (self::$buffer) {
            $model = $this->getModel();
            self::$buffer->put(BufferedWriter::STATEMENT_INSERT_ODKU, get_class($model), $model->getTable(), $values);
            return true;
        } else {
            return parent::update($values);
        }
    }
    
    /**
     * Update record(s) in the database, but only allow them to be updated whtn the builder is buffered
     *
     * @param  array  $values
     * @return int
     */
    public function insertUpdate(array $values)
    {
        if (!self::$buffer) {
            throw new \Exception('BufferedBuilder::insertUpdate can only be used with BufferedWriter!');
        }
        
        $model = $this->getModel();
        self::$buffer->put(BufferedWriter::STATEMENT_INSERT_ODKU, get_class($model), $model->getTable(), $values);
        return true;
    }
    
    /**
     * New buffering session (preserving previous if any)
     * 
     * @param \Closure|callable $callable
     */
    public static function bufferedSession($callable)
    {
        $customWriter = new BufferedWriter();
        $prevBuffer = static::$buffer;
        static::$buffer = $customWriter;
        
        $returnValue = $callable();
        
        $customWriter->flushAll();
        static::$buffer = $prevBuffer;
        
        return $returnValue;
    }
}
