<?php

namespace InSegment\ApiCore\Services;

use Illuminate\Support\Arr;

use InSegment\ApiCore\Services\MicroEvent;

/**
 * Bufferizes Model writes based on statement (INSERT, REPLACE etc), their classes then tables
 * and flushed by chunks
 */
class BufferedWriter
{
    const STATEMENT_INSERT = 0;
    const STATEMENT_INSERT_ODKU = 1;
    const STATEMENT_REPLACE = 2;
    
    /**
     * Amount of records of some class to be buffered before bulk write occurs
     *
     * @var int
     */
    private $bufferSizeInRecords;
    
    /**
     * Buffers for all Model classes
     * 
     * @var array 
     */
    private $buffers = [];
    
    /**
     * Incrementing id of BufferedWriter
     *
     * @var int
     */
    private static $incrementId;
    private $id;
    
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->bufferSizeInRecords = config('api_core.buffer_size');
        $this->id = self::$incrementId++;
    }
    
    /**
     * Get id of BufferedWriter
     * 
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }
    
    /**
     * Put values for one Model
     * 
     * @param int $statement
     * @param string $modelClass
     * @param string $modelTable
     * @param array $values
     * @return null
     */
    public function put(int $statement, string $modelClass, string $modelTable, array $values)
    {
        if (empty($values)) {
            return;
        }
        
        $keys = array_keys($values);
        
        if (!isset($this->buffers[$statement][$modelClass][$modelTable])) {
            $this->buffers[$statement][$modelClass][$modelTable] = ['keys' => $keys, 'values' => [$values], 'count' => 0];
            $currentBuffer = $this->buffers[$statement][$modelClass][$modelTable];
        } else {
            $currentBuffer = &$this->buffers[$statement][$modelClass][$modelTable];
            
            if ($keys !== $currentBuffer['keys']) {
                $currentBuffer['keys'] = array_unique(array_merge($keys, $currentBuffer['keys']));
            }
            
            $currentBuffer['values'][] = $values;
            ++$currentBuffer['count'];
        }
        
        if ($currentBuffer['count'] === $this->bufferSizeInRecords) {
            $this->flush($statement, $modelClass, $modelTable);
        }
    }
    
    /**
     * Flush all stored values
     * 
     * @return null
     */
    public function flushAll()
    {
        foreach ($this->buffers as $statement => $buffersByStatemet) {
            foreach ($buffersByStatemet as $modelClass => $buffersByClass) {
                foreach (array_keys($buffersByClass) as $modelTable) {
                    $this->flush($statement, $modelClass, $modelTable);
                }
            }
        }
    }
    
    /**
     * Flush one particular statement of one Model class and one table
     * 
     * @param int $statement
     * @param string $modelClass
     * @param string $modelTable
     * @return null
     */
    protected function flush(int $statement, string $modelClass, string $modelTable)
    {
        $currentBuffer = $this->buffers[$statement][$modelClass][$modelTable];
        
        if (empty($currentBuffer)) {
            return;
        }
        
        $baseBuilder = $modelClass::query()->toBase();
        $baseBuilder->from = $modelTable;
        
        $keys = $currentBuffer['keys'];
        sort($keys);
        $substitutes = array_combine($keys, array_fill(0, count($keys), null));
        foreach ($currentBuffer['values'] as &$buffered) {
            $buffered += $substitutes;
            ksort($buffered);
        }
        
        $insertSql = $baseBuilder->grammar->compileInsert($baseBuilder, $currentBuffer['values']);
        
        switch($statement) {
            case self::STATEMENT_INSERT:
                $querySql = $insertSql;
            break;
            case self::STATEMENT_REPLACE:
                $querySql = 'REPLACE' . substr($insertSql, 6);
            break;
            case self::STATEMENT_INSERT_ODKU:
                $odkuKeys = [];
                
                foreach ($currentBuffer['keys'] as $key) {
                    $odkuKeys[] = "`{$key}` = VALUES(`{$key}`)";
                }
                
                $odku = ' ON DUPLICATE KEY UPDATE ' . implode(',', $odkuKeys);
                $querySql = $insertSql.$odku;
            break;
        }
        
        $baseBuilder->connection->affectingStatement($querySql, Arr::flatten($currentBuffer['values'], 1));
        unset($this->buffers[$statement][$modelClass][$modelTable]);
        MicroEvent::fireEvent(static::class, 'flushed', $this, $modelClass);
    }
}
