<?php

namespace InSegment\ApiCore\Services\DiffSlice;

use Illuminate\Support\Facades\DB;
use InSegment\ApiCore\Models\SliceStructure;
use InSegment\ApiCore\Models\UUIDGeneration;
use InSegment\ApiCore\Models\UUIDReserve;

class DiffSlice
{
    
    /**
     * Reserve of supplementary keys
     *
     * @var \InSegment\ApiCore\Models\UUIDReserve
     */
    protected $uuidReserve;
    
    /**
     * Query for marking UUID's as used
     *
     * @var \InSegment\ApiCore\Services\BufferedBuilder
     */
    protected $uuidLogQuery;
    
    /**
     * Instance of SliceStructure
     *
     * @var \InSegment\ApiCore\Models\SliceStructure 
     */
    protected $sliceStructure;
    
    /**
     * Enumeration of all tables used in operation
     *
     * @var string[]
     */
    protected $allTargetTables;
    
    /**
     * Temporary tables of operation
     *
     * @var array [
     *      string $tableName => [
     *          string $type => string $tempTableOfType
     *      ],
     *      ...
     * ],
     */
    protected $tempTables;
    
    /**
     * UID of operation
     *
     * @var string
     */
    protected $uid;
    
    /**
     * Counters for written and new records in the Slice (for estimates)
     *
     * @var array [
     *     'new' => [
     *         string $table => int,
     *         ...
     *     ]
     *     'written' => [
     *         string $table => int,
     *         ...
     *     ]
     * ] 
     */
    protected $counters = [];
    
    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Models\SliceStructure  $sliceStructure
     * @param array $allTargetTables
     */
    public function __construct(SliceStructure $sliceStructure, array $allTargetTables)
    {
        $this->uid = $sliceStructure->getUid();
        $this->uuidReserve = new UUIDReserve($this->uid);
        $this->uuidLogQuery = (new UUIDGeneration([], $this->uid))->newQuery();
        $this->sliceStructure = $sliceStructure;
        $this->allTargetTables = $allTargetTables;
        $this->tempTables = $sliceStructure->getTablesInSchema();
    }
    
    /**
     * Establish necessary preparations
     * 
     * @param array $counts
     * @param bool $inMemory
     */
    public function establish(array $counts, bool $inMemory = false)
    {
        UUIDGeneration::establish([
            'uid' => $this->uid,
            'tables' => $this->allTargetTables,
            'inMemory' => $inMemory
        ]);

        foreach ($counts as $table => $amount) {
            $this->uuidReserve->allocate($table, $amount);
        }
    }
    
    /**
     * Get reserve of supplementary keys
     *
     * @return \InSegment\ApiCore\Models\UUIDReserve
     */
    public function getUuidReserve(): UUIDReserve
    {
        return $this->uuidReserve;
    }
    
    /**
     * Query for marking UUID's as used
     *
     * @return \InSegment\ApiCore\Services\BufferedBuilder
     */
    public function getUuidLogQuery()
    {
        return $this->uuidLogQuery;
    }
    
    /**
     * Get temporary tables for target table
     * 
     * @param string $targetTable
     * @return array [
     *     string $type => string $tempTableOfType
     * ],
     * @throws \Exception
     */
    public function getTempTables(string $targetTable): array
    {
        if (!isset($this->tempTables[$targetTable])) {
            throw new \Exception("Temporary tables are missing for: {$targetTable}");
        }
        
        return $this->tempTables[$targetTable];
    }
    
    /**
     * Give a new importer for the specified class
     * 
     * @param string $class
     * @return \InSegment\ApiCore\Services\DiffSlice\ImportClass
     */
    public function importClass(string $class): ImportClass
    {
        return new ImportClass($this, $class);
    }
    
    /**
     * Drop temporary tables
     * 
     * @return null
     */
    public function dropTempTables()
    {
        foreach ($this->tempTables as $tempTables) {
            foreach ($tempTables as $tempTable) {
                list ($tempTableDb, $tempTableName) = explode('.', $tempTable);
                DB::statement("DROP TABLE IF EXISTS `{$tempTableDb}`.`{$tempTableName}`");
            }
        }
        
        UUIDGeneration::disband($this->uid);
    }
    
    /**
     * Increment counter for new rows in table
     * 
     * @param string $table
     * @return null
     */
    public function incrementNewCount(string $table)
    {
        if (!isset($this->counters['new'][$table])) {
            $this->counters['new'][$table] = 1;
        } else {
            ++$this->counters['new'][$table];
        }
    }
    
    /**
     * Increment counter for written rows in table
     * 
     * @param string $table
     * @return null
     */
    public function incrementWrittenCount(string $table)
    {
        if (!isset($this->counters['written'][$table])) {
            $this->counters['written'][$table] = 1;
        } else {
            ++$this->counters['written'][$table];
        }
    }
    
    /**
     * Get counters for written and new records for estimates
     *
     * @return array [
     *     'new' => [
     *         string $table => int,
     *         ...
     *     ]
     *     'written' => [
     *         string $table => int,
     *         ...
     *     ]
     * ] 
     */
    public function getCounters(): array
    {
        return $this->counters;
    }
    
}
