<?php

namespace InSegment\ApiCore\Services;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;

/**
 * Uses tricks and dances around fire to bring effect of 'inverseOf' relation
 * Not very stable, do not use in common business logic
 */
class JoinResolver
{
    /**
     * Enabled or disabled
     * 
     * @var bool 
     */
    public static $enabled = false;
    
    /**
     * Tables grimouire
     * Does not expect sudden changes
     * 
     * @var array
     */
    private static $tables = [];
    
    /**
     * Calculates join keys and sets them as info on the related model of left relation
     * 
     * @param \Illuminate\Database\Eloquent\Relations\Relation $leftRelation
     * @param string $rightRelationName
     * @return null
     */
    public static function resolve(Relation $leftRelation, string $rightRelationName)
    {
        if (!static::$enabled) {
            return;
        }
        
        $leftParent = $leftRelation->getParent();
        $leftRelated = $leftRelation->getRelated();
        $leftTable = $leftParent->getTable();
        $rightTable = $leftRelated->getTable();
        
        if (!isset(self::$tables[$leftTable][$rightTable])) {
            $rightRelation = $leftRelated->$rightRelationName();
            $concreteKeys = $rightRelation->relationJoinKeys($leftParent);
            self::$tables[$leftTable][$rightTable] = static::unqualify($concreteKeys);
        }
        
        $inverseRelationData = &$leftRelated->accessInverseRelations();
        $inverseRelationData['keys'][$rightRelationName] = self::$tables[$leftTable][$rightTable];
        $inverseRelationData['values'][$rightRelationName] = $leftParent;
    }
    
    /**
     * Used as a hack in the Model, transfers common key from to left model to proper attribute of the right model,
     * and sets a relation. Used when the Model is newFromBuilder
     * 
     * @param \Illuminate\Database\Eloquent\Model $leftModel
     * @param \Illuminate\Database\Eloquent\Model $rightModel
     * @return null
     */
    public static function apply(Model $leftModel, Model $rightModel)
    {
        if (!static::$enabled) {
            return;
        }
        
        $inverseRelationData = &$leftModel->accessInverseRelations();
        foreach ($inverseRelationData['values'] as $relationName => $model) {
            $keys = $inverseRelationData['keys'][$relationName];
            if (static::transferBetween($model, $rightModel, $keys)) {
                $rightModel->setRelation($relationName, $model);
            }
        }
    }
    
    /**
     * Another hack in the Model, transfers from the right model to the left model proper attribute and sets a relation,
     * used when the relation with left model is set on right model.
     * Thus, no check for the common key to match: we know they are related
     * 
     * As models loaded by eager load are not discarded automatically, we need to clear info about inverse on them,
     * so it won't affect subsequent loads
     * 
     * @param \Illuminate\Database\Eloquent\Model $leftModel
     * @param \Illuminate\Database\Eloquent\Model $rightModel
     * @return null
     */
    public static function applyEager(Model $leftModel, Model $rightModel)
    {
        if (!static::$enabled) {
            return;
        }
        
        $inverseRelationData = &$leftModel->accessInverseRelations();
        foreach ($inverseRelationData['values'] as $relationName => $model) {
            if (get_class($model) == get_class($rightModel)) {
                $keys = $inverseRelationData['keys'][$relationName];
                if (static::transferBetween($rightModel, $leftModel, $keys, true)) {
                    $leftModel->setRelation($relationName, $rightModel);
                }
                
                unset($inverseRelationData['values'][$relationName]);
                unset($inverseRelationData['keys'][$relationName]);
            }
        }
        
        if (empty($inverseRelationData['values'])) {
            unset($inverseRelationData['values'], $inverseRelationData['keys']);
            $inverseRelationData = null;
        }
    }
    
    /**
     * Strips table name and dots of keys
     * 
     * @param array $keys
     * @return array
     */
    public static function unqualify(array &$keys): array
    {
        foreach ($keys as &$pair) {
            foreach (['left', 'right'] as $which) {
                $pair[$which] = last(explode('.', $pair[$which]));
            }
        }
        
        return $keys;
    }
    
    /**
     * Copy common attribute
     * In the eager mode, does not check if the target model already have one (replaces)
     * 
     * @param Model $source
     * @param Model $target
     * @param array $keys
     * @param bool $eager
     * @return boolean
     */
    protected static function transferBetween(Model $source, Model $target, array $keys, bool $eager = false)
    {
        foreach ($keys as $pair) {
            if (!$eager && ($targetLeft = $target->getAttribute($pair['left'])) !== null) {
                if ($targetLeft != $source->getAttribute($pair['right'])) {
                    return false;
                }
            } else {
                $target->setAttribute($pair['left'], $source->getAttribute($pair['right']));
            }
        }
        
        return true;
    }
}