<?php

namespace InSegment\ApiCore\Services;

use Soundasleep\Html2Text;

/**
 * Class Logger
 *
 * @author khartakhai@infuseua.com
 * @package App
 */
final class Logger
{

    /**
     * @var string
     */
    protected $logPath = '';

    /**
     * @var string
     */
    protected $logFile = 'api-log.json';

    /**
     * @var string
     */
    protected $errorFile = 'api-error.json';

    /**
     * @var string
     */
    protected $logHistory = 'loghistory.txt';

    /**
     * @var array [[string $date, string $type, array|string $data], ...]
     */
    protected $log = [];
    
    /**
     * @var string|null
     */
    protected $lastLog;

    /**
     * @var string
     */
    protected $writtenLogFile = '';

    /**
     * @var array [string $file, string $file, ...]
     */
    protected $writtenErrorFiles = [];

    /**
     * Creates an instance of Logger.
     * 
     * @param string $logPath    [Optional] Path to log files.
     * @param string $logFile    [Optional] File in JSON format.
     * @param string $errorFile  [Optional] File in HTML format.
     * @return Logger
     */
    public function __construct(string $logPath = '', string $logFile = '', string $errorFile = '')
    {
        $this->setLogPath($logPath);
        $this->setLogFile($logFile);
        $this->setErrorFile($errorFile);
    }

    /**
     * 
     * @param string $path Path to log files.
     * @return \self
     */
    public function setLogPath(string $path): self
    {
        if (!empty($path)) {
            $path = realpath($path);

            if (!$path) {
                $logsStorageBasePath = config(
                    'app.logs_storage_path',
                    app('path.storage') . DIRECTORY_SEPARATOR . 'logs'
                );
                $path = realpath($logsStorageBasePath);
            }

            $path .= $path ? DIRECTORY_SEPARATOR : '';

            $this->logPath = $path;
        }
        return $this;
    }

    /**
     * 
     * @param string $file File for log output in json format.
     * @return \self
     */
    public function setLogFile(string $file): self
    {
        if (!empty($file)) {
            $this->logFile = $file;
        }
        return $this;
    }

    /**
     * 
     * @param string $file File for error output.
     * @return \self
     */
    public function setErrorFile(string $file): self
    {
        if (!empty($file)) {
            $this->errorFile = $file;
        }
        return $this;
    }

    /**
     * 
     * @return string
     */
    public function getLogPath()
    {
        return $this->logPath;
    }

    /**
     * 
     * @return string
     */
    public function getLogFile()
    {
        return $this->logFile;
    }

    /**
     * 
     * @return string
     */
    public function getErrorFile()
    {
        return $this->errorFile;
    }

    /**
     * Get the filename of the log, that was written.
     *
     * @return string
     */
    public function getWrittenLogFile()
    {
        return $this->writtenLogFile;
    }

    /**
     * Get the filenames of the errors, that were written.
     *
     * @return array [string $file, string $file, ...]
     */
    public function getWrittenErrorFiles()
    {
        return $this->writtenErrorFiles;
    }

    /**
     * 
     * @param string $tag
     * @param array|string $data
     * @return \self
     */
    public function push(string $tag, $data): self
    {
        if (is_string($data)) {
            if ($this->errorFile) {
                $file = $this->uniqueFilename($this->errorFile);
                $dataText = Html2Text::convert($data, ['ignore_errors' => true]);
                $this->write($file, json_encode($dataText, JSON_PRETTY_PRINT|JSON_PARTIAL_OUTPUT_ON_ERROR));
                $fileName = basename($file);
                $data = "See error file: {$fileName}";
                $this->writtenErrorFiles[] = $fileName;
            }
        }

        return $this->pushString($tag, json_encode($data, JSON_PRETTY_PRINT|JSON_PARTIAL_OUTPUT_ON_ERROR));
    }
    
    /**
     * 
     * @param string $tag
     * @param string $data
     * @return \self
     */
    public function pushString(string $tag, string $data)
    {
        $date = (new \DateTime())->format('Y-m-d H:i:s.u');
        $tag = "{$date} > {$tag}";
        $n = 1;

        while (isset($this->log["{$tag} ({$n})"])) {
            $n ++;
        }
        
        if (isset($this->lastLog)) {
            $this->log[$this->lastLog] .= ',';
        }

        $logKey = "{$tag} ({$n})";
        $this->log[$logKey] = $data;
        $this->lastLog = $logKey;
        return $this;
    }

    /**
     * Flushes log and error in respective files.
     */
    public function close()
    {
        if (!empty($this->log) && $this->logFile) {
            $file = $this->uniqueFilename($this->logFile);
            $toImplode = ["{\n"];
            foreach ($this->log as $logKey => $logData) {
                $toImplode[] = "    \"{$logKey}\": ";
                $string = is_string($logData) ? $logData : json_encode($logData, JSON_PRETTY_PRINT|JSON_PARTIAL_OUTPUT_ON_ERROR);
                $toImplode[] = implode("\n    ", explode("\n", $string));
                $toImplode[] = "\n";
            }
            $toImplode[] = "}";
            
            $this->write($file, implode('', $toImplode));
            $this->writtenLogFile = basename($file);
        }
        $this->log = [];
        $this->lastLog = null;

        $cwd = getcwd();
        chdir($this->logPath);
        $files = implode("\n", glob("*.api-*"));
        chdir($cwd);

        $this->write("{$this->logPath}{$this->logHistory}", $files);
    }

    /**
     * 
     * @param string $filename
     * @param string $content
     */
    private function write(string $filename, string $content)
    {
        file_put_contents($filename, $content);
    }

    /**
     * 
     * @param string $filename
     * @return string Returns "/path/to/YYYYMMDD.NUMBER.filename"
     */
    private function uniqueFilename(string $filename): string
    {
        $date = (new \DateTime)->format('Ymd');
        $commonPart = "{$this->logPath}{$date}";
        $list = glob("{$commonPart}.*");
        $increment = count($list) + 1;

        while (true) {
            $file = sprintf("{$commonPart}.%04d.{$filename}", $increment);

            if (!file_exists($file)) {
                return $file;
            }

            $increment ++;
        }
    }

}
