<?php

namespace InSegment\ApiCore\Services;

use InSegment\ApiCore\Traits\RunsProcessesTrait;
use Illuminate\Support\Facades\DB;

/**
 * @see https://php.net/manual/ru/filters.compression.php for compressing resource streams
 */
class MysqlCommands
{
    use RunsProcessesTrait;
    
    /**
     * @var array
     */
    protected $config;
    
    /**
     * Constructor
     * 
     * @param array $config [
     *     'username' => string,
     *     'password' => string,
     *     'host' => string,
     *     'port' => int,
     * ]
     */
    public function __construct(array $config)
    {
        $this->config = $this->validateConfig($config);
    }
    
    /**
     * Make MysqlCommands from Laravel/Lumen connection config
     * 
     * @return \static
     * @throws \Exception
     */
    public static function fromConnectionConfig()
    {
        $connectionName = DB::connection()->getName();
        $databaseConfig = config("database.connections.{$connectionName}");

        if (!$databaseConfig) {
            throw new \Exception("Database config cannot be detected");
        }
        
        return new static($databaseConfig);
    }
    
    /**
     * mysql command
     * 
     * @param string $schema
     * @param resource|string|\Symfony\Component\Process\InputStream $input
     * @param callable|null $callback
     * @return int|null
     * @throws \Exception
     */
    public function mysql(string $schema, $input, $callback = null)
    {
        if (!$schema) {
            throw new \Exception("Schema name must not be empty");
        }
        
        $commandName = 'mysql';
        $arguments = [
            '--user=' => $this->config['username'],
            '--password=' => $this->config['password'],
            '--host=' => $this->config['host'],
            '--port=' => $this->config['port'],
            $schema,
        ];
        
        $process = $this->getProcess($commandName, $arguments);
        $process->setInput($input);
        return $this->processWithInput($process, $callback);
    }
    
    /**
     * mysqldump command
     * 
     * @param string $schema
     * @param array $tables
     * @param resource|null $outputResource
     * @return string|null
     * @throws \Exception
     */
    public function mysqldump(string $schema, array $tables, $outputResource = null)
    {
        if (!$schema) {
            throw new \Exception("Schema name must not be empty");
        }
        
        if (!count($tables)) {
            throw new \Exception("Dumping the whole schema is not allowed, please provide specific tables");
        }
        
        $commandName = 'mysqldump';
        $arguments = [
            '--user=' => $this->config['username'],
            '--password=' => $this->config['password'],
            '--host=' => $this->config['host'],
            '--port=' => $this->config['port'],
            $schema,
        ];
        
        foreach ($tables as $table) {
            if (!is_string($table) || !$table) {
                throw new \Exception("All table names should be non-empty strings");
            }

            $arguments[] = $table;
        }
        
        $process = $this->getProcess($commandName, $arguments);
        return $this->processWithOutput($process, $outputResource);
    }
    
    /**
     * Validation of configuration
     * 
     * @param array $config
     * @return array
     * @throws \Exception
     */
    protected function validateConfig(array $config)
    {
        if (($missing = array_diff(['username', 'password', 'host', 'port', 'database'], array_keys($config)))) {
            $implodeMissing = implode(', ', $missing);
            throw new \Exception("Missing necessary fields in config: {$implodeMissing}");
        }
        
        $errors = [];
        foreach (['username', 'host'] as $param) {
            if (!is_string($config[$param]) || !$config[$param]) {
                $errors[] = "{$param} is not a string or empty";
            }
        }
        
        if (!is_string($config['password'])) {
            $errors[] = "password is not a string";
        }
        
        if (!is_int($config['port']) && !ctype_digit($config['port']) || $config['port'] < 1) {
            $errors[] = "{$param} is not a positive integer";
        }
        
        if (count($errors)) {
            $implodeFails = implode(', ', $errors);
            throw new \Exception("Configuration does not pass validation: {$implodeFails}");
        }
        
        return $config;
    }

}
